Option Strict Off
Option Explicit On
Imports System.Runtime.InteropServices
Module MCUG3API
   '  +-----------------------------------------------------------------------+
   '  | (C) Roesch & Walter Industrie-Elektronik GmbH                         |
   '  | Woertelweg 2 b/c             D-77963 Schwanau                         |
   '  +-------------------------------+---------------------------------------+
   '  | Tel : +49 (0) 7824/6628-0     | email    : support@rw-gmbh.de         |
   '  | Fax : +49 (0) 7824/6628-29    | Internet : http://www.rw-gmbh.de      |
   '  +-------------------------------+---------------------------------------+
   '  | Project     : MCUG3 TOOLSET   | Compiler : VisualBasic.NET            |
   '  |                               |                                       |
   '  | Module name : mcug3.vb        | Operating System:                     |
   '  |                               |            - Windows 95/98/Me         |
   '  |                               |            - Windows NT 4.0           |
   '  |                               |            - Windows NT Embedded 4.0  |
   '  |                               |            - Windows 2000             |
   '  |                               |            - Windows XP               |
   '  |                               |            - Windows Vista            |
   '  |                               |            - Windows 7                |
   '  |                               |            - Windows 8                |
   '  |                               |            - Windows 10               |
   '  +-------------------------------+---------------------------------------+
   '  | Author : Roesch R.                                                    |
   '  |          Walter M.                         07.10.2005                 |
   '  +-----------------------------------------------------------------------+
   '  | Description   : Driver Interface for                                  |
   '  |                 High-Performance Motion-Control Family Generation 3   |
   '  |                                                                       |
   '  |                 Currently supported G3-Controllers:                   |
   '  |                         MCU-3000, APCI-8001, MCU-6000 and APCI-8401   |
   '  |                         MCU-3100, APCI-8008, MCU-3106 and APCI-8006   |
   '  |                                                                       |
   '  |                 This file contains all prototypes, definitions and    |
   '  |                 structures to build a Win32-based application program |
   '  |                                                                       |
   '  +-----------------------------------------------------------------------+

   '    Revision History:
   '    -----------------
   '    04/08/2020  V2-53-130 mw new function ClearCI added
   '    22/06/2020  V2-53-130 mw function added: CleanServiceUsers
   '    27/09/2016  V2-53-113 mw functions added: wrSampleTimeExpansion and rdSampleTimeExpansion
   '    25/07/2016  V2-53-113 mw new functions: sstvl, spda, spdr
   '    18/03/2016  V2-53-111 mw new functions: wrShortProfile and rdShortProfile
   '    18/02/2016  V2-53-110 mw new functions: rdVBError, wrVBError, SendReqScannerBuffer, InitMcuSystem4, wrtrac, wrtrvl, wrtrtvl, rdtrac, rdtrvl, rdtrtvl
   '                             new prototype for rddigof - command can be used as fast alternative for rddigo
   '                             but is working only for the first axis of every axis group
   '                             eg. MCU-3000 axis index must be 0, 3 or 6
   '    12/02/2014  V2-53-91 mw  new function rdpeaux
   '    12/02/2014  V2-53-91 mw  BoardTypes supplemented
   '    14/09/2012  V2-53-86 mw  BoardTypes added
   '    15/03/2011  V2-53-79 mw	new function RPtoDP
   '    26/10/2005  V2-53-34 mw  Structure Alignment rectified
   '                             parameters of ja and jr changed
   '    19/10/2005  V2-53-34 mw  new function: rdptp, rdpir, rddtv
   '    13/10/2005  V2-53-33 mw	new functions: rdtrjerk, wrtrjerk
   '    07/10/2005  V2-53-33 mw  created from VB6 mcug3.bas
   '    23/03/2007  V2-53-46 mw  parameter changed at functions
   '        							rdrv, getcnctsrclinenr, getmsgstr, getcnctfh, burnaplpar, gettimers, cm
   '                					Attention: incompatibillity is possible
   '    10.12.2007  V2-53.56 mw  New Functions wrtrovrst, rdrtrovrst added
   '        							for handling of trovr settling time
   '    12.12.2007	V2-53.56 mw	rdifs now returns zero, if function sucessfully has been executet
   '        							up to now rdifs hasn't any return value
   '    04/01/2008	V2-53-57 mw	new functions rdgfaux and wrgfaux
   '
   '    Hints:
   '     Structure CTRLRPARAMS is not functioning up to now
   '     feedback is welcome to info@rw-gmbh.de
   '

   Public Const MAXAXIS As Short = 18 ' currently maximum supplied axis
   Public Const SIZE_STRREV As Short = 40 ' String-Size definition
   Public Const SIZE_CIBUF As Short = 1000 ' Number of elements in common (long) integer buffer
   Public Const SIZE_CDBUF As Short = 1000 ' Number of elements in common double buffer
   Public Const BSIZE As Short = 1024 ' Size of System File Header and CNC File Header
   Public Const SIZE_FILENAMES As Short = 256 ' Max. (char) Size for Filenames
   Public Const SIZE_TIME As Short = 32 ' Size (char) of System time information
   Public Const SCPSZ_R As Short = 15 ' row size for CTRLRPARAMS
   Public Const SCPSZ_C As Short = 15 ' colomn size for CTRLRPARAMS
   Public Const SPD2D_S As Short = 10 ' size of SPD2D
   ' Bit-Locations for Status-und Error-Flags in axst '
   ' error flags ...
   Public Const BITLOC_to As Short = &H1S ' Bit 0:  MCU-6000, APCI-8401 only, MCU-3000 / APCI-8001 = 0
   Public Const BITLOC_eo As Short = &H2S ' Emergency Out
   Public Const BITLOC_dnr As Short = &H4S ' Drive not ready
   Public Const BITLOC_lslh As Short = &H8S ' Limit switch left hardware
   Public Const BITLOC_lsrh As Short = &H10S ' Limit switch right hardware
   Public Const BITLOC_lsls As Short = &H20S ' Limit switch left software
   Public Const BITLOC_lsrs As Short = &H40S ' Limit switch right software
   Public Const BITLOC_mpe As Short = &H80S ' Maximum position error
   Public Const BITLOC_dhef As Short = &H100S ' Data handling error flag
   Public Const BITLOC_cef As Short = &H200S ' Configuration error flag
   ' Bit 10..11: currently not used
   ' status flags ...
   Public Const BITLOC_pe As Short = &H1000S ' Profile end
   Public Const BITLOC_cl As Short = &H2000S ' Closed loop
   Public Const BITLOC_ip As Short = &H4000S ' In position
   Public Const BITLOC_ui As Short = &H8000S ' user input
   Public Const BITLOC_lpsf As Integer = &H10000 ' latched position synchronous flag
   ' Bit 17..29: currently not used
   ' Bit 30..31: for internal use only

   ' Enumeration for Position units
   Enum PUS
      MM
      INCH
      M
      REV
      DEG
      RAD
      COUNTS
      STEPS
   End Enum

   ' Enumeration for Time units
   Enum TUS
      SEC
      Min
      TSAMPLE
   End Enum

   ' Enumeration for MCU-G3 Board Types
   Enum BOARDTYPE
      UNKNOWNBOARD   = &H0S 'unknown board type
      MCU3000        = &H4S 'G3-Controller MCU-3000
      APCI8001       = &H4S 'G3-Controller APCI-8001
      MCU6000        = &H8S 'G3-Controller MCU-6000
      APCI8401       = &H8S 'G3-Controller APCI-8401
      MCU3400        = &H10S 'G3-Controller MCU-3400
      CPCI8001       = &H10S 'G3-Controller CPCI-8001
      MCU3100        = &H20S 'G3-Controller MCU-3100
      APCI8008       = &H20S 'G3-Controller APCI-8008
      MCU3106        = &H40S 'G3-Controller MCU-3106
      APCI8006       = &H40S 'G3-Controller APCI-8006
   End Enum

   ' Enumeration for AT Access
   Enum ATAccessType
      ATAccessNone
      ATAccessInput
      ATAccessOutput
      ATAccessInputOutput
   End Enum
    
   Enum ATDataType
      ATDataNone
      ATDataByte
      ATDataWord
      ATDataDoubleWord
      ATDataReal
      ATDataSingle
   End Enum

   Enum ATBusNr
      CanOpenBus     = 400
      InterBus       = 500
      ResourcenBus   = 1000
      ScannerBus     = 1100
      ElCamBus       = 1200
      TcBus          = 1300
   End Enum

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure TSRP
        ' Common-Data for DLL-interface-driver MCUG3.DLL and PC-Application-Program(s)
        <FieldOffset(0)> Dim an As Integer 'Axis-Number
        <FieldOffset(4)> Dim kp As Double 'PIDF filter parameter kp
        <FieldOffset(12)> Dim ki As Double 'PIDF filter parameter ki
        <FieldOffset(20)> Dim kd As Double 'PIDF filter parameter kd
        <FieldOffset(28)> Dim kpl As Double 'PIDF filter parameter kpl
        <FieldOffset(36)> Dim kfca As Double 'PIDF forward compensation acceleration
        <FieldOffset(44)> Dim kfcv As Double 'PIDF forward compensation velocity
        <FieldOffset(52)> Dim jac As Double 'jog acceleration
        <FieldOffset(60)> Dim jvl As Double 'jog velocity
        <FieldOffset(68)> Dim jtvl As Double 'target velocity
        <FieldOffset(76)> Dim jovr As Double 'jog override
        <FieldOffset(84)> Dim hac As Double 'home acceleration
        <FieldOffset(92)> Dim hvl As Double 'home velocity
        <FieldOffset(100)> Dim rp As Double 'real position
        <FieldOffset(108)> Dim dp As Double 'desired position
        <FieldOffset(116)> Dim tp As Double 'target position
        <FieldOffset(124)> Dim dv As Double 'desired velocity
        <FieldOffset(132)> Dim lp As Double 'latched position
        <FieldOffset(140)> Dim aux As Double 'auxiliary value
        <FieldOffset(148)> Dim sll As Double 'software limit left
        <FieldOffset(156)> Dim slr As Double 'software limit right
        <FieldOffset(164)> Dim ipw As Double 'in position window
        <FieldOffset(172)> Dim mpe As Double 'maximum position error
        <FieldOffset(180)> Dim sdec As Double 'stop deceleration
        <FieldOffset(188)> Dim gf As Double 'gear factor
        <FieldOffset(196)> Dim mcp As Integer 'motor command port
        <FieldOffset(200)> Dim axst As Integer 'axis status
        <FieldOffset(204)> Dim lsm As Integer 'left spool memory
        <FieldOffset(208)> Dim epc As Integer 'eeprom programming cycle
        <FieldOffset(212)> Dim digi As Integer 'dig. inputs = asmi (MCU-6000, APCI-8401)
        <FieldOffset(216)> Dim digo As Integer 'dig. outputs = asmo (MCU-6000, APCI-8401)
        <FieldOffset(220)> Dim ifs As Integer 'interface status = asms (MCU-6000, APCI-8401)

        <FieldOffset(224)> Dim scratch1 As Integer
        <FieldOffset(228)> Dim scratch2 As Integer
        <FieldOffset(232)> Dim scratch3 As Integer
        <FieldOffset(236)> Dim scratch4 As Integer
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure ASEL
        ' Axis-Selection
        <FieldOffset(0)> Dim unoa As Integer ' Number of used axis
        '<VBFixedArray(MAXAXIS - 1)> Dim san() As Integer ' Axis Numbers
        <FieldOffset(4), VBFixedArray(MAXAXIS - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=MAXAXIS)> Dim san() As Integer ' Axis Numbers

        'UPGRADE_TODO: for inizialisation of instances of this structur "Initialize" must be called. More information's under 'ms-help://MS.VSCC/commoner/redir/redirect.htm?keyword="vbup1026"'
        Public Sub Initialize()
            ReDim san(MAXAXIS - 1)
        End Sub
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure tru
        ' Trajectory Units
        <FieldOffset(0)> Dim pu As Integer ' position unit, see enumeration PUS above
        <FieldOffset(4)> Dim tu As Integer ' time unit, see enumeration PUS above
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure lmp
        ' Linear Motion Profile parameters
        <FieldOffset(0)> Dim ac As Double ' trajectory acceleration
        <FieldOffset(8)> Dim vl As Double ' trajectory velocity
        <FieldOffset(16)> Dim tvl As Double ' trajectory target velocity
        ' <VBFixedArray(MAXAXIS - 1)> Dim dtm() As Double ' distance to move
        <FieldOffset(24), VBFixedArray(MAXAXIS - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=MAXAXIS)> Dim dtm() As Double ' distance to move

        'UPGRADE_TODO: for inizialisation of instances of this structur "Initialize" must be called. More information's under 'ms-help://MS.VSCC/commoner/redir/redirect.htm?keyword="vbup1026"'
        Public Sub Initialize()
            ReDim dtm(MAXAXIS - 1)
        End Sub
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure CMP
        ' Circular Motion Profile parameters
        <FieldOffset(0)> Dim ac As Double ' trajectory acceleration
        <FieldOffset(8)> Dim vl As Double ' trajectory velocity
        <FieldOffset(16)> Dim tvl As Double ' trajectory target velocity
        <FieldOffset(24)> Dim phi As Double ' angel phi
        <FieldOffset(32)> Dim dtca1 As Double ' center point of axis 1
        <FieldOffset(40)> Dim dtca2 As Double ' center point of axis 2
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure HMP
        'Helical Motion Profile parameters
        <FieldOffset(0)> Dim ac As Double ' trajectory acceleration
        <FieldOffset(8)> Dim vl As Double ' trajectory velocity
        <FieldOffset(16)> Dim tvl As Double ' trajectory target velocity
        <FieldOffset(24)> Dim phi As Double ' angel phi
        <FieldOffset(32)> Dim dtca1 As Double ' center point of axis 1
        <FieldOffset(40)> Dim dtca2 As Double ' center point of axis 2
        '<VBFixedArray(MAXAXIS - 1)> Dim dtm() As Double ' distance to move axis 3..n
        <FieldOffset(48), VBFixedArray(MAXAXIS - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=MAXAXIS)> Dim dtm() As Double ' distance to move axis 3..n

        'UPGRADE_TODO: for inizialisation of instances of this structur "Initialize" must be called. More information's under 'ms-help://MS.VSCC/commoner/redir/redirect.htm?keyword="vbup1026"'
        Public Sub Initialize()
            ReDim dtm(MAXAXIS - 1)
        End Sub
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure HMP3D
        ' 3D Helical Motion Profile parameters
        <FieldOffset(0)> Dim ac As Double ' trajectory acceleration
        <FieldOffset(8)> Dim vl As Double ' trajectory velocity
        <FieldOffset(16)> Dim tvl As Double ' trajectory target velocity
        <FieldOffset(24)> Dim phi As Double ' angel phi
        <FieldOffset(32)> Dim dtca1 As Double ' center point of axis 1
        <FieldOffset(40)> Dim dtca2 As Double ' center point of axis 2
        <FieldOffset(48)> Dim dtca3 As Double ' center point of axis 3
        <FieldOffset(56)> Dim pn1 As Double   ' area perpendicular of axis 1 (Flchen-Normale A1)
        <FieldOffset(64)> Dim pn2 As Double   ' area perpendicular of axis 2 (Flchen-Normale A2)
        <FieldOffset(72)> Dim pn3 As Double   ' area perpendicular of axis 3 (Flchen-Normale A3)
        '<VBFixedArray(MAXAXIS - 1)> Dim dtm() As Double ' distance to move axis 3..n
        <FieldOffset(80), VBFixedArray(MAXAXIS - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=MAXAXIS)> Dim dtm() As Double ' distance to move axis 3..n

        'UPGRADE_TODO: for inizialisation of instances of this structur "Initialize" must be called. More information's under 'ms-help://MS.VSCC/commoner/redir/redirect.htm?keyword="vbup1026"'
        Public Sub Initialize()
            ReDim dtm(MAXAXIS - 1)
        End Sub
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure CNCTS
        ' CNC-Task Status variables
        <FieldOffset(0)> Dim errnum As Integer ' error number
        <FieldOffset(4)> Dim errline As Integer ' error line
        <FieldOffset(8)> Dim stackfree As Integer ' stack bytes free
        <FieldOffset(12)> Dim running As Integer ' nc task running status
        <FieldOffset(16)> Dim csrcline As Integer ' current (processed) source line number
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure ROSI
        ' RISC Operating System Informations
        ' <VBFixedArray(SIZE_STRREV - 1)> Dim revision() As Byte
        <FieldOffset(0), VBFixedArray(SIZE_STRREV - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=SIZE_STRREV)> Dim revision() As Byte
        ' software-revision-string
        <FieldOffset(SIZE_STRREV)> Dim number_axis As Integer  ' number of supplied axis
        <FieldOffset(SIZE_STRREV + 4)> Dim sysfile_loaded As Integer ' status flag for sysfile already loaded

        'UPGRADE_TODO: for inizialisation of instances of this structur "Initialize" must be called. More information's under 'ms-help://MS.VSCC/commoner/redir/redirect.htm?keyword="vbup1026"'
        Public Sub Initialize()
            ReDim revision(SIZE_STRREV - 1)
        End Sub
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure CIBUF
        ' Buffer definition for common (long) integer variables
        '<VBFixedArray(SIZE_CIBUF - 1)> Dim civals() As Integer
        <FieldOffset(0), VBFixedArray(SIZE_CIBUF - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=SIZE_CIBUF)> Dim civals() As Integer

        'UPGRADE_TODO: for inizialisation of instances of this structur "Initialize" must be called. More information's under 'ms-help://MS.VSCC/commoner/redir/redirect.htm?keyword="vbup1026"'
        Public Sub Initialize()
            ReDim civals(SIZE_CIBUF - 1)
        End Sub
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure CDBUF
        ' Buffer definition for common double variables
        ' <VBFixedArray(SIZE_CDBUF - 1)> Dim cdvals() As Double
        <FieldOffset(0), VBFixedArray(SIZE_CDBUF - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=SIZE_CDBUF)> Dim cdvals() As Double

        'UPGRADE_TODO: for inizialisation of instances of this structur "Initialize" must be called. More information's under 'ms-help://MS.VSCC/commoner/redir/redirect.htm?keyword="vbup1026"'
        Public Sub Initialize()
            ReDim cdvals(SIZE_CDBUF - 1)
        End Sub
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure CBCNCT
        ' Common CNC-Task buffer
        <FieldOffset(0)> Dim TaskNr As Integer ' Task Number 0..3
        <FieldOffset(4)> Dim size As Integer ' size of Buffer [bytes]
        <FieldOffset(8)> Dim BuffPtr() As Integer ' pointer to user allocated buffer
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure SPDBUF
        ' position data to spool
        <FieldOffset(0)> Dim dtm As Double ' distance to move
    End Structure


    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure SPVTDBUF
        ' position, velocity, time datas to spool
        <FieldOffset(0)> Dim dtm As Double ' distance to move
        <FieldOffset(8)> Dim vl As Double ' velocity
        <FieldOffset(16)> Dim t As Double ' time
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure CTRLRPARAMS
        ' Controller specific data fields (application specicfic)
        'UPGRADE_NOTE: val wurde aktualisiert auf value. Klicken Sie hier fr weitere Informationen: 'ms-help://MS.VSCC/commoner/redir/redirect.htm?keyword="vbup1061"'
        '<VBFixedArray(SCPSZ_R - 1, SCPSZ_C - 1)> Dim value(,) As Double
        <FieldOffset(0), VBFixedArray((SCPSZ_R - 1), (SCPSZ_C - 1)), MarshalAs(UnmanagedType.ByValArray, SizeConst:=(SCPSZ_R * SCPSZ_C))> Dim value(,) As Double
        'UPGRADE_TODO: for inizialisation of instances of this structur "Initialize" must be called. More information's under 'ms-help://MS.VSCC/commoner/redir/redirect.htm?keyword="vbup1026"'
        Public Sub Initialize()
            ReDim value(SCPSZ_R - 1, SCPSZ_C - 1)
        End Sub
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure SPD2DPARAMS
        <FieldOffset(0)> Dim size As Integer ' size
        '<VBFixedArray(1)> Dim san() As Integer ' selected axis numbers
        <FieldOffset(4), VBFixedArray(1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=2)> Dim san() As Integer ' selected axis numbers
        '<VBFixedArray(SPD2D_S - 1)> Dim dtma1() As Double ' distance to move axis 1
        <FieldOffset(12), VBFixedArray(SPD2D_S - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=SPD2D_S)> Dim dtma1() As Double ' distance to move axis 1
        '<VBFixedArray(SPD2D_S - 1)> Dim dtma2() As Double ' distance to move axis 2
        <FieldOffset(12 + (SPD2D_S * 8)), VBFixedArray(SPD2D_S - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=SPD2D_S)> Dim dtma2() As Double ' distance to move axis 2

        'UPGRADE_TODO: for inizialisation of instances of this structur "Initialize" must be called. More information's under 'ms-help://MS.VSCC/commoner/redir/redirect.htm?keyword="vbup1026"'
        Public Sub Initialize()
            ReDim san(1)
            ReDim dtma1(SPD2D_S - 1)
            ReDim dtma2(SPD2D_S - 1)
        End Sub
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure LATCH_INFOS
        ' parameter set for latching position datas
        <FieldOffset(0)> Dim san As Integer ' selected axis number
        <FieldOffset(4)> Dim lst_ As Double ' latch start time
        'UPGRADE_NOTE: let und lst wurden aktualisiert auf let_ und lst_. Klicken Sie hier fr weitere Informationen: 'ms-help://MS.VSCC/commoner/redir/redirect.htm?keyword="vbup1061"'
        <FieldOffset(12)> Dim let_ As Double ' latch end time
        <FieldOffset(20)> Dim pu As Integer ' position unit
        <FieldOffset(24)> Dim tu As Integer ' time unit
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure OLSR_INFOS
        ' parameter set for open loop system response
        <FieldOffset(0)> Dim san As Integer ' selected axis number
        <FieldOffset(4)> Dim delay As Double ' delay [s]
        <FieldOffset(12)> Dim mcp As Double ' motor command value [V]
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure FILEHEADER
        ' CNC Task File Header Information
        <FieldOffset(0)> Dim TaskNr As Integer ' NC-File runs in task 'TaskNr' 0..3
        '<VBFixedArray(SIZE_FILENAMES - 1)> Dim SrcFileName() As Byte
        <FieldOffset(4), VBFixedArray(SIZE_FILENAMES - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=SIZE_FILENAMES)> Dim SrcFileName() As Byte
        ' CNC Source File Name
        '<VBFixedArray(SIZE_FILENAMES - 1)> Dim CncFileName() As Byte
        <FieldOffset(4 + SIZE_FILENAMES), VBFixedArray(SIZE_FILENAMES - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=SIZE_FILENAMES)> Dim CncFileName() As Byte
        ' CNC pseudo code file name
        '<VBFixedArray(SIZE_STRREV - 1)> Dim RevNr() As Byte
        <FieldOffset(4 + SIZE_FILENAMES + SIZE_FILENAMES), VBFixedArray(SIZE_STRREV - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=SIZE_STRREV)> Dim RevNr() As Byte
        ' revision number
        '<VBFixedArray(SIZE_TIME - 1)> Dim DateTimearray() As Byte
        <FieldOffset(4 + SIZE_FILENAMES + SIZE_FILENAMES + SIZE_STRREV), VBFixedArray(SIZE_TIME - 1), MarshalAs(UnmanagedType.ByValArray, SizeConst:=SIZE_TIME)> Dim DateTimearray() As Byte
        ' build date and time information

        'UPGRADE_TODO: for inizialisation of instances of this structur "Initialize" must be called. More information's under 'ms-help://MS.VSCC/commoner/redir/redirect.htm?keyword="vbup1026"'
        Public Sub Initialize()
            ReDim SrcFileName(SIZE_FILENAMES - 1)
            ReDim CncFileName(SIZE_FILENAMES - 1)
            ReDim RevNr(SIZE_STRREV - 1)
            ReDim DateTimearray(SIZE_TIME - 1)
        End Sub
    End Structure

    <StructLayout(LayoutKind.Sequential, Pack:=1)> Structure OptionDescriptorObject
        <FieldOffset(0)> Dim Handle As Integer ' do not change that!
        <FieldOffset(4)> Dim AccessType As Integer ' Read, Write, Flag
        <FieldOffset(8)> Dim DataType As Integer ' Byte, Word, Double Word, Real, ...
        <FieldOffset(12)> Dim BusNumber As Integer ' Interbus, Canbus, G3-Resourcebus, Scannerbus ...
        <FieldOffset(16)> Dim DeviceNumber As Integer ' depends on Bus Type
        <FieldOffset(20)> Dim Index As Integer ' depends on Bus Type
        <FieldOffset(24)> Dim SubIndex As Integer ' depends on Bus Type
    End Structure


   '  +-----------------------------------------------------------------------+
   '  | Prototypes standard functions,                                        |
   '  | functions are described in MCU-G3 Programming Manual                  |
   '  +-----------------------------------------------------------------------+

   Declare Sub uf Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'update filter
   Declare Sub ctru Lib "MCUG3.DLL" (ByRef DTRU As tru) 'change trajectory parameters
   Declare Sub wrjac Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write jog acceleration
   Declare Sub wrjvl Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write jog velocity
   Declare Sub wrjtvl Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write jog target velocity
   Declare Sub wrhac Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write home acceleration
   Declare Sub wrhvl Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write home velocity
   Declare Sub wrrp Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write real position
   Declare Sub wrdp Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write desired position
   Declare Sub wrlp Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write latched position
   Declare Sub wraux Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write auxiliary value
   Declare Sub shp Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'set home position
   Declare Sub wrdigo Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-3000, APCI-8001 only: write digital outputs
   Declare Sub wrdigob Lib "MCUG3.DLL" (an As Integer, bitnr As Integer, value As Integer)
                                                         'MCU-3000, APCI-8001 only: write digital output bit
   Declare Sub wrasmo Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-6000, APCI-8401 only: write ASM digital outputs
   Declare Sub wrasmob Lib "MCUG3.DLL" (an As Integer, bitnr As Integer, value As Integer)
                                                         'MCU-6000, APCI-8401 only: write ASM digital output bit
   Declare Sub rdrp Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read real position
   Declare Sub rddp Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read desired position
   Declare Sub rdtp Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read target position
   Declare Sub rddv Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read desired velocity
   Declare Sub rdlp Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read latched position
   Declare Sub rdaux Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read auxiliary value
   Declare Sub rdauxf Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read auxiliary value fast
   Declare Sub rddigi Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-3000, APCI-8001 only: read digital inputs
   Declare Function rddigib Lib "MCUG3.DLL" (an As Integer, bitnr As Integer) As Integer
                                                         'MCU-3000, APCI-8001 only: read digital input bit
   Declare Function rdifs Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) As Integer 'MCU-3000, APCI-8001 only: read interface status
   Declare Function rdifsb Lib "MCUG3.DLL" (an As Integer, bitnr As Integer) As Integer
                                                         'MCU-3000, APCI-8001 only: read interface status bit
   Declare Sub rddigo Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-3000, APCI-8001 only: read digital outputs
   Declare Sub rddigof Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) ' MCU-3000, APCI-8001 only: read digital outputs direct: respect comment in header
   Declare Function rddigob Lib "MCUG3.DLL" (an As Integer, bitnr As Integer) As Integer
                                                         'MCU-3000, APCI-8001 only: read digital output bit
   Declare Sub rdasmi Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-6000, APCI-8401 only: read ASM digital inputs
   Declare Function rdasmib Lib "MCUG3.DLL" (an As Integer, bitnr As Integer) As Integer
                                                         'MCU-6000, APCI-8401 only: read ASM digital input bit
   Declare Sub rdasms Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-6000, APCI-8401 only: read ASM status
   Declare Function rasmsb Lib "MCUG3.DLL" (an As Integer, bitnr As Integer) As Integer
                                                         'MCU-6000, APCI-8401 only: read ASM status bit
   Declare Sub rdepc Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-6000, APCI-8401 only: read eeprom programming cycle
   Declare Sub rdasmepc Lib "MCUG3.DLL" (ByRef DTSRP As TSRP)
                                                         'MCU-6000, APCI-8401 only: read ASM eeprom programming cycle
   Declare Sub rdasmo Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-6000, APCI-8401 only: read ASM digital outputs
   Declare Function rdasmob Lib "MCUG3.DLL" (an As Integer, bitnr As Integer) As Integer
                                                         'MCU-6000, APCI-8401 only: read ASM digital output bit
   Declare Sub rdlsm Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read left spool memory
   Declare Sub rdaxst Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read axis status
   Declare Sub rdaxstf Lib "MCUG3.DLL" (ByRef DTSRP As TSRP)  'read axis status fast
   Declare Function rdaxstb Lib "MCUG3.DLL" (an As Integer, bitnr As Integer) As Integer
                                                         'read axis status bit
   Declare Sub rdjac Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read jog acceleration
   Declare Sub rdjvl Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read jog velocity
   Declare Sub rdjtvl Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read jog target velocity
   Declare Sub rdhac Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read home acceleration
   Declare Sub rdhvl Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read home velocity
   Declare Sub cl  Lib "MCUG3.DLL" (ByRef DASEL As ASEL) 'close loop
   Declare Sub clv Lib "MCUG3.DLL" (ByRef DASEL As ASEL) 'close loop velocity
   Declare Sub ol Lib "MCUG3.DLL" (ByRef DASEL As ASEL) 'open loop
   Declare Sub ra Lib "MCUG3.DLL" (ByRef DASEL As ASEL) 'reset axis
   Declare Sub rifs Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-3000, APCI-8001 only: reset interface status flags
   Declare Sub rasms Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-6000, APCI-8401 only: reset ASM status flags
   Declare Sub rs Lib "MCUG3.DLL" () 'reset system
   Declare Sub azo Lib "MCUG3.DLL" (set_ As Integer)
                                                           ' activate zero offset
   Declare Sub jr Lib "MCUG3.DLL" (ByRef DASEL As ASEL, <[In](), Out()> DTSRP() As TSRP)
                                                           ' jog relative
   Declare Sub ja Lib "MCUG3.DLL" (ByRef DASEL As ASEL, <[In](), Out()> DTSRP() As TSRP)
                                                           ' jog absolute
   Declare Sub jhl Lib "MCUG3.DLL" (ByRef DASEL As ASEL)   ' jog home left
   Declare Sub jhr Lib "MCUG3.DLL" (ByRef DASEL As ASEL)   ' jog home right
   Declare Sub jhi Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef TSRP As TSRP)
                                                         'jog relative (to) index
   Declare Sub mlr Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef lmp As lmp)
                                                         ' move linear relative
   Declare Sub mla Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef lmp As lmp)
                                                         ' move linear absolute
   Declare Sub mcr Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef CMP As CMP)
                                                         ' move circular relative
   Declare Sub mca Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef CMP As CMP)
                                                         ' move circular absolute
   Declare Sub mhr Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef HMP As HMP)
                                                         ' move helix relative
   Declare Sub mha Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef HMP As HMP)
                                                         ' move helix absolute
   Declare Sub mca3d Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef HMP3D As HMP3D)
                                                         ' move circular 3D absolute
   Declare Sub mcr3d Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef HMP3D As HMP3D)
                                                         ' move circular 3D relative
   Declare Sub smlr Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef lmp As lmp)
                                                         ' spool motion linear relative
   Declare Sub smlri Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef lmp As lmp, PcapIndex As Integer)
                                                         'spool motion linear relative indexed
   Declare Sub smla Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef lmp As lmp)
                                                         ' spool motion linear absolute
   Declare Sub smlai Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef lmp As lmp, PcapIndex As Integer)
                                                         'spool motion linear absolute indexed
   Declare Sub smcr Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef CMP As CMP)
                                                         ' spool motion circular relative
   Declare Sub smca Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef CMP As CMP)
                                                         ' spool motion circular absolute
   Declare Sub smhr Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef HMP As HMP)
                                                         ' spool motion helix relative
   Declare Sub smha Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef HMP As HMP)
                                                         ' spool motion helix absolute
   Declare Sub smca3d Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef HMP3D As HMP3D)
                                                         ' spooled move circular 3D absolute
   Declare Sub smcr3d Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef HMP3D As HMP3D)
                                                         ' spooled move circular 3D relative
   Declare Sub ssms Lib "MCUG3.DLL" (ByRef DASEL As ASEL) 'start spooled motions synchronous
   Declare Sub sstps Lib "MCUG3.DLL" (ByRef DASEL As ASEL) 'stop spooled motions synchronous
   Declare Sub sdels Lib "MCUG3.DLL" (ByRef DASEL As ASEL) 'delete spooled motions synchronous
   Declare Function txbf Lib "MCUG3.DLL" (filename As String) As Integer
                                                         'transmit binary file
   Declare Function txbf2 Lib "MCUG3.DLL" (filename As String) As Integer
                                                         'transmit binary file, 2nd method
   Declare Sub startcnct Lib "MCUG3.DLL" (TaskNr As Integer)
                                                         'start cnc task
   Declare Sub stopcnct Lib "MCUG3.DLL" (TaskNr As Integer)
                                                         'stop cnc task
   Declare Sub contcnct Lib "MCUG3.DLL" (TaskNr As Integer)
                                                         'continue cnc task
   Declare Sub stepcnct Lib "MCUG3.DLL" (TaskNr As Integer)
                                                         'step cnc task
   Declare Sub rdcncts Lib "MCUG3.DLL" (TaskNr As Integer, ByRef CNCTS As CNCTS)
                                                         'read cnc task status
   Declare Function rdcbcnct Lib "MCUG3.DLL" (ByRef DCBCNCT As CBCNCT) As Integer
                                                         'read cnc task spec. common buffer
   Declare Function wrcbcnct Lib "MCUG3.DLL" (ByRef DCBCNCT As CBCNCT) As Integer
                                                         'write cnc task spec. common buffer
   Declare Sub rdci Lib "MCUG3.DLL" (ndx As Integer, ByRef CIBUF As CIBUF)
                                                         'read common integer
   Declare Sub rdcd Lib "MCUG3.DLL" (ndx As Integer, ByRef CDBUF As CDBUF)
                                                         'read common double
   Declare Sub wrci Lib "MCUG3.DLL" (ndx As Integer, ByRef CIBUF As CIBUF)
                                                         'write common integer variables
   Declare Sub wrci_ Lib "MCUG3.DLL" (ndx As Integer, ByRef CIBUF As CIBUF)
                                                         'write common integer variables per Communication
   Declare Sub ClearCI99 Lib "MCUG3.DLL" ()               'Clear CI99 per Communication
   Declare Sub ClearCI   Lib "MCUG3.DLL" (ndx As Integer) 'Clear CI indexed per Communication
   
   Declare Sub wrcd Lib "MCUG3.DLL" (ndx As Integer, ByRef CDBUF As CDBUF)
                                                         'write common double variables
   Declare Sub rdmcp Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read motor command port
   Declare Sub wrmcp Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write motor command port
   Declare Sub wrledrd Lib "MCUG3.DLL" (value As Integer)
                                                         'write led red
   Declare Sub wrledyl Lib "MCUG3.DLL" (value As Integer)
                                                         'write led yellow
   Declare Sub wrledgn Lib "MCUG3.DLL" (value As Integer)
                                                         'read led green
   Declare Function rdledrd Lib "MCUG3.DLL" () As Integer 'read led red
   Declare Function rdledyl Lib "MCUG3.DLL" () As Integer 'read led yellow
   Declare Function rdledgn Lib "MCUG3.DLL" () As Integer 'read led green
   Declare Function rdirqpc Lib "MCUG3.DLL" () As Integer 'read irq pc
   Declare Sub mcuinit Lib "MCUG3.DLL" (ByRef DROSI As ROSI) 'MCU-G3 initialisition
   Declare Sub mcuinit2 Lib "MCUG3.DLL" (ByRef DROSI As ROSI, TpuBaseAddress As Integer)
                                                          'MCU-G3 initialisition, 2nd method
   Declare Sub wrjovr Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write jog override
   Declare Sub rdjovr Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read jog override
   Declare Sub wrtrovr Lib "MCUG3.DLL" (ByRef value As Double) 'write trajectory override
   Declare Sub rdtrovr Lib "MCUG3.DLL" (ByRef value As Double) 'read trajectory override
   Declare Function wrtrovrst Lib "MCUG3.DLL" (ByRef value As Double) As Integer 'write trajectory override settling time
   Declare Function rdtrovrst Lib "MCUG3.DLL" (ByRef value As Double) As Integer 'read trajectory override settling time
   Declare Function wrShortProfile Lib "MCUG3.DLL" (ByRef value As Double) As Integer 'write Short Profile Threshold for NoTriangle
   Declare Function rdShortProfile Lib "MCUG3.DLL" (ByRef value As Double) As Integer 'read Short Profile Threshold for NoTriangle
   Declare Sub utrovr Lib "MCUG3.DLL" (ByRef DASEL As ASEL) 'update trajectory override
   Declare Sub rdf Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read filter
   Declare Sub rdap Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read axis parameters
   Declare Sub rdsll Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read software limit left
   Declare Sub wrsll Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write software limit left
   Declare Sub rdslr Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read software limit right
   Declare Sub wrslr Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write software limit right
   Declare Sub rdipw Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read in position window
   Declare Sub wripw Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write in position window
   Declare Sub rdmpe Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read maximum position error
   Declare Sub wrmpe Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write maximum position error
   Declare Sub rdsdec Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read stop deceleration
   Declare Sub wrsdec Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write stop deceleration
   Declare Sub spd Lib "MCUG3.DLL" (an As Integer, size As Integer, ByRef SPDBUF As SPDBUF)  'spool position data
   Declare Sub spda Lib "MCUG3.DLL" (an As Integer, size As Integer, ByRef SPDBUF As SPDBUF, PcapIndex As Integer)   'spool position data absolut indicated
   Declare Sub spdr Lib "MCUG3.DLL" (an As Integer, size As Integer, ByRef SPDBUF As SPDBUF, PcapIndex As Integer)   'spool position data relative indicated
   Declare Sub sstvl Lib "MCUG3.DLL" (ByRef DASEL As ASEL, tvl As Double)    'spooler set target velocity

   Declare Sub wrgf Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'write gear factor
   Declare Sub rdgf Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'read gear factor
   Declare Sub spvtd Lib "MCUG3.DLL" (an As Integer, size As Integer, ByRef SPVTDBUF As SPVTDBUF)
                                                         'spool position, velocity, time data
   Declare Sub ms Lib "MCUG3.DLL" (ByRef DASEL As ASEL) 'motion stop
   Declare Sub js Lib "MCUG3.DLL" (ByRef DASEL As ASEL) 'jog stop
   Declare Sub lps Lib "MCUG3.DLL" (an As Integer, mst As Integer)
                                                         'latch position synchronous
   Declare Sub rdigi Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-3000, APCI-8001 only: reset digital inputs
   Declare Sub rdlpndx Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-3000, APCI-8001 only: read latched position index
   Declare Sub wrlpndx Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) 'MCU-3000, APCI-8001 only: write latched position index
   Declare Sub ssf Lib "MCUG3.DLL" (an As Integer, cmd As Integer, value As Double)          ' Spooler-Special-Function-Command
   Declare Sub ssfn Lib "MCUG3.DLL" (ByRef DASEL As ASEL, command As Integer, Value As Double)     ' Spooler-Special-Function-Command for more than one axis
   Declare Sub ssfi Lib "MCUG3.DLL" (an As Integer, command As Integer, Value As Double, PcapIndex As Integer)      ' Spooler-Special-Function-Command
   Declare Sub ssfni Lib "MCUG3.DLL" (ByRef DASEL As ASEL, command As Integer, Value As Double, PcapIndex As Integer)      ' Spooler-Special-Function-Command for more than one axis indicated

   Declare Function InitMcuSystem Lib "MCUG3.DLL" (<[In](), Out()> DTSRP() As TSRP) As Integer
                                                         'initialisation function first method, (you should not use this function, instead use InitMcuSystem2() or even better InitMcuSystem3())
   Declare Function InitMcuSystem2 Lib "MCUG3.DLL" (<[In](), Out()> DTSRP() As TSRP, TpuBaseAddress As Integer, filename As String) As Integer
                                                         'initialisation function, 2nd method
   Declare Function InitMcuSystem3 Lib "MCUG3.DLL" (<[In](), Out()> DTSRP() As TSRP, ByRef DROSI As ROSI, TpuBaseAddress As Integer, filename As String, ByRef BDTYPE As BOARDTYPE) As Integer
                                                         'initialisation function, 3nd method
   Declare Function InitMcuSystem4 Lib "MCUG3.DLL" (<[In](), Out()> DTSRP() As TSRP, ByRef DROSI As ROSI, TpuBaseAddress As Integer, filename As String, ByRef BDTYPE As BOARDTYPE, ByRef NrCards As Integer) As Integer
    ' initialisation function, 4th method (method 3 with additional number of cards result)
   Declare Sub InitMcuErrorReport Lib "MCUG3.DLL" (errnum As Integer)
                                                         'detailed initialisation error report function
   Declare Sub txbfErrorReport Lib "MCUG3.DLL" (filename As String, errnum As Integer)
                                                         'detailed load error report function
   Declare Sub BootErrorReport Lib "MCUG3.DLL" (filename As String, errnum As Integer)
                                                         'detailed boot error report function
   Declare Sub McuClose Lib "MCUG3.DLL" ()               'close driver function, must be called at end of program !
   Declare Function BootFile Lib "MCUG3.DLL" (filename As String, TpuBaseAddress As Integer) As Integer
                                                         'Send Bootfile (i.e. rwmos.elf) to controller board

   Declare Sub LdSSplineA Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef lmp As lmp)
                                                         ' Load Spool Spline Data Absolut
   Declare Sub LdSSplineR Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef lmp As lmp)
                                                         ' Load Spool Spline Data Relativ
   Declare Sub ExeSSpline Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef lmp As lmp)
                                                         ' Execute Spool Spline

   Declare Function wrOptionInt Lib "MCUG3.DLL" (ByRef odesc As OptionDescriptorObject, ByRef value As Integer) As Integer
                                                         ' Option Object write integer function
   Declare Function rdOptionInt Lib "MCUG3.DLL" (ByRef odesc As OptionDescriptorObject, ByRef value As Integer) As Integer
                                                         ' Option Object read integer function
   Declare Function wrOptionDbl Lib "MCUG3.DLL" (ByRef odesc As OptionDescriptorObject, ByRef value As Double) As Integer
                                                         ' Option Object write double function
   Declare Function rdOptionDbl Lib "MCUG3.DLL" (ByRef odesc As OptionDescriptorObject, ByRef value As Double) As Integer
                                                         ' Option Object read double function
   Declare Function freeOptionHandle Lib "MCUG3.DLL" (ByRef odesc As OptionDescriptorObject) As Integer
                                                         ' Option Object free handle

   Declare Function rdScannerBufferSize Lib "MCUG3.DLL" () As Integer
                                                         ' Read size of scan buffer
   Declare Function rdScannerStatus Lib "MCUG3.DLL" () As Integer
                                                         ' read scanner status register
   Declare Function rdScannerLsm Lib "MCUG3.DLL" () As Integer
                                                         ' read scanner left spool memory
   Declare Function rdScannerBuffer Lib "MCUG3.DLL" (ByRef buffer As String, size As Integer) As Integer           ' read scanner buffer
   Declare Function SendReqScannerBuffer Lib "MCUG3.DLL" (ByRef buffer As String, size As Integer) As Integer      ' read scanner buffer by Send Request
   Declare Sub RPtoDP Lib "MCUG3.DLL" (ByRef DASEL As ASEL) ' Real Position to Desired Position


   '  +---------------------------------------------------------------------------------+
   '  |    Prototypes for client and application specific functions,                    |
   '  |    do not use when you have no information about the individual functions !!!   |
   '  |    (normaly you should not need that functions)                                 |
   '  +---------------------------------------------------------------------------------+

   Declare Sub wrBlk Lib "MCUG3.DLL" (ByRef buffer As String) ' Common Write Command without return value (internal function)
   Declare Sub rdBlk Lib "MCUG3.DLL" (ByRef buffer As String) ' Common Read Command
   Declare Sub wrtp Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) ' write target position
   Declare Sub rdrpd Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) ' read real position display
   Declare Sub rddpd Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) ' read desired position display
   Declare Sub rdtpd Lib "MCUG3.DLL" (ByRef DTSRP As TSRP) ' read target position display
   Declare Sub scp Lib "MCUG3.DLL" (an As Integer, ByRef DCTRLRPARAMS As CTRLRPARAMS)
                                                         ' set controller parameters
   Declare Sub rdrv Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                         ' read real velocity
   Declare Sub cm Lib "MCUG3.DLL" (an As Integer, ByRef value As Integer)
                                                         ' change mode
   Declare Sub spd2d Lib "MCUG3.DLL" (ByRef DSPD2DPARAMS As SPD2DPARAMS)
                                                         ' spool position datas 2D
   Declare Sub rdlrpr Lib "MCUG3.DLL" (ByRef DLATCH_INFOS As LATCH_INFOS, ByRef buffer As String)
                                                         ' read latched real pos. registers
   Declare Sub rdldpr Lib "MCUG3.DLL" (ByRef DLATCH_INFOS As LATCH_INFOS, ByRef buffer As String)
                                                         ' read latched desired pos. registers
   Declare Sub getcnctsrclinenr Lib "MCUG3.DLL" (TaskNr As Integer, ByRef LineNr As Integer)
                                                         ' read in current linenumber beeing processed of selected cnc task
   Declare Function getmsgstr Lib "MCUG3.DLL" (an As Integer, ByRef result As String) As Integer
                                                         ' read in internal MCU-G3 debug-string
   Declare Sub getcnctfh Lib "MCUG3.DLL" (TaskNr As Integer, ByRef DFILEHEADER As FILEHEADER)
                                                         ' read in CNC-task-fileheader of selected cnc task
   Declare Sub burnaplpar Lib "MCUG3.DLL" (an As Integer) ' burn appliance parameters
   Declare Sub gettimers Lib "MCUG3.DLL" (ByRef DTSRP As TSRP, an As Integer)
                                                         ' read in several debug-timers of high-prior task
   Declare Sub lpr Lib "MCUG3.DLL" (ByRef LATCH_INFOS As LATCH_INFOS)
                                                         ' latch position registers
   Declare Sub lprs Lib "MCUG3.DLL" (ByRef DASEL As ASEL, ByRef DLATCH_INFOS As LATCH_INFOS)
                                                         ' latch position registers synchronous on several axis
   Declare Sub olsr Lib "MCUG3.DLL" (ByRef DOLSR_INFOS As OLSR_INFOS)
                                                         ' open loop system response
   Declare Sub rdasmecs Lib "MCUG3.DLL" (ByRef DTSRP As TSRP, ByRef an As Integer)
                                                         ' MCU-6000, APCI-8401 only: read in ASM eprom checksum of firmware
   Declare Sub ReadLinkStatus Lib "MCUG3.DLL" (ByRef fae As Integer, ByRef ba As Integer)
                                                         ' read current link stati
   Declare Function rdAin Lib "MCUG3.DLL" (an As Integer, channel As Integer) As Integer
                                                         ' MCU-3000, APCI-8001 only: read Analog Inputs from OPMF-3001
   Declare Sub szpa Lib "MCUG3.DLL" (an As Integer, Position As Double)
                                                         ' Set Zero Position absolut
   Declare Sub szpr Lib "MCUG3.DLL" (an As Integer, Position As Double)
                                                         ' Set Zero Position relativ
   Declare Sub rdMcuStatus Lib "MCUG3.DLL" (ByRef buffer As String, size As Integer)
                                                         ' read in MCU G3 status registers
   Declare Sub rdAsmSecureParams Lib "MCUG3.DLL" (an As Integer, ByRef buffer As String)
                                                         ' MCU-6000, APCI-8401 only: read ASM-EEPROM parameters (firmware data)
   Declare Sub wrAsmSecureParams Lib "MCUG3.DLL" (an As Integer, ByRef buffer As String)
                                                         ' MCU-6000, APCI-8401 only: write ASM-EEPROM parameters (firmware data)
'---------------------------------------------------------
   Declare Sub wrModeReg Lib "MCUG3.DLL" (ByRef value As Integer)
                                                         ' write Mode Register
   Declare Sub rdModeReg Lib "MCUG3.DLL" (ByRef value As Integer)
                                                         'read Mode Register
   Declare Sub wrnfrax Lib "MCUG3.DLL" (ByRef value As Integer)
                                                          'write No-Feed-Rate-Axis
   Declare Sub rdnfrax Lib "MCUG3.DLL" (ByRef value As Integer)
                                                          'read No-Feed-Rate-Axis
   Declare Sub wrErrorReg Lib "MCUG3.DLL" (ByRef value As Integer)
                                                          'write Error Register
   Declare Sub rdErrorReg Lib "MCUG3.DLL" (ByRef value As Integer)
                                                          'read Error Register
   Declare Sub wrMDVel Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                          'write Maximum Delta Velocity at axis
   Declare Sub rdMDVel Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                          'read Maximum Delta Velocity from axis
   Declare Sub wrJerkRel Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                          'write jerkrel at axis
   Declare Sub rdJerkRel Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                          'read jerkrel from axis
   Declare Sub wrMaxVel Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                          'write Maximum Axis Velocity at axis
   Declare Sub rdMaxVel Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                          'read Maximum Axis Velocity from axis
   Declare Sub wrMaxAcc Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                          'write Maximum Axis Acceleration at axis
   Declare Sub rdMaxAcc Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                          'read Maximum Axis Acceleration from axis
   Declare Sub wrEffRadius Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                          'write Effective Radius for rotatoric Axis
   Declare Sub rdEffRadius Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                          'read Effective Radius for rotatoric Axis
   Declare Sub rdPosErr Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                          'read Position Error from axis
   Declare Sub rdMCiS Lib "MCUG3.DLL" (an As Integer, ByRef value As Integer)
                                                          'read Move Commands in Spooler from axis
   Declare Sub MsgToScreen Lib "MCUG3.DLL" (Enable As Integer)
                                                          'enable / disable screen messages
   Declare Function CardSelect Lib "MCUG3.DLL" (CardNr As Integer) As Integer
                                                          ' read in internal MCU-G3 debug-string
   Declare Function IntToPointer Lib "MCUG3.DLL" (ByRef val As Integer) As Integer
                                                          ' get the address of a Integer
   Declare Function DblToPointer Lib "MCUG3.DLL" (ByRef val As Double) As Integer
                                                          ' get the address of a double
   Declare Function CharToPointer Lib "MCUG3.DLL" (ByRef val As Byte) As Integer
                                                          ' get the address of a char
   Declare Sub wrslsp Lib "MCUG3.DLL" (an As Integer, ByRef value As Double)
                                                          'write slsp
   Declare Function rdslsp Lib "MCUG3.DLL" (an As Integer, ByRef value As Double) As Integer
                                                          'read slsp
   Declare Function accstop Lib "MCUG3.DLL" (ByRef DASEL As ASEL) As Integer
                                                          'acceleration stop in JOG-S-Profiles
   Declare Function getEnvStr Lib "MCUG3.DLL" (EnvStr As String) As Integer
                                                          ' read Environment String
   Declare Function gettskinfo Lib "MCUG3.DLL" (tasknr As Integer, ByRef tskinfo As Integer) As Integer
                                                          ' read Environment String
   Declare Function gettskstr Lib "MCUG3.DLL" (tasknr As Integer, buffer As String, szbuffer As Integer) As Integer
                                                          ' read Environment String
   'Declare Function MCUG3_ResetBoardIntRoutine Lib "MCUG3.DLL" () As Integer
                                                          ' deinstall interrupt handler
   'Declare Function MCUG3_SetBoardIntRoutine Lib "MCUG3.DLL" (func As Integer) As Integer
                                                          ' install interrupt handler
   Declare Function wrdpoffset Lib "MCUG3.DLL" (an As Integer, ByRef value As Double) As Integer
                                                          'write dpoffset to axis
   Declare Function rddpoffset Lib "MCUG3.DLL" (an As Integer, ByRef value As Double) As Integer
                                                          'read dpoffset from axis
   Declare Function wrdvoffset Lib "MCUG3.DLL" (an As Integer, ByRef value As Double) As Integer
                                                           'write dvoffset to axis
   Declare Function rddvoffset Lib "MCUG3.DLL" (an As Integer, ByRef value As Double) As Integer
                                                            'read dvoffset from axis
   Declare Function rdZeroOffset Lib "MCUG3.DLL" (an As Integer, ByRef value As Double) As Integer
                                                          'read  absolute Value of actual set Zero Offset
   Declare Function wrtrjerk Lib "MCUG3.DLL" (ByRef value As Double) As Integer ' write trajectory override
   Declare Function rdtrjerk Lib "MCUG3.DLL" (ByRef value As Double) As Integer ' read trajectory override
   Declare Function rdptp    Lib "MCUG3.DLL" (an As Integer, ByRef value As Double) As Integer
                                                                                ' read  profile target position
   Declare Function rdpir Lib "MCUG3.DLL" (ByRef value As Integer) As Integer   ' read  Profile Information Register
   Declare Function rddtv Lib "MCUG3.DLL" (<[In](), Out()> ByRef value() As Double) As Integer    ' read  desired trajectory velocity

   Declare Function wrSplineBlockSize Lib "MCUG3.DLL" (ByRef value As Integer) As Integer       ' write SplineBlockSize
   Declare Function rdSplineBlockSize Lib "MCUG3.DLL" (ByRef value As Integer) As Integer       ' read  SplineBlockSize
   Declare Function wrgfaux Lib "MCUG3.DLL" (an As Integer, ByRef value As Double) As Integer   'write Auxiliary Gear Factor to axis
   Declare Function rdgfaux Lib "MCUG3.DLL" (an As Integer, ByRef value As Double) As Integer   'read Auxiliary Gear Factor from axis
   Declare Function rdpeaux Lib "MCUG3.DLL" (an As Integer, ByRef value As Double) As Integer   'read rp-aux difference from axis
   Declare Function wrtrac Lib "MCUG3.DLL" (ByRef value As Double) As Integer ' write trajectory acceleration
   Declare Function wrtrvl Lib "MCUG3.DLL" (ByRef value As Double) As Integer ' write trajectory velocity
   Declare Function wrtrtvl Lib "MCUG3.DLL" (ByRef value As Double) As Integer ' write trajectory target velocity
   Declare Function rdtrac Lib "MCUG3.DLL" (ByRef value As Double) As Integer ' read trajectory acceleration
   Declare Function rdtrvl Lib "MCUG3.DLL" (ByRef value As Double) As Integer ' read trajectory velocity
   Declare Function rdtrtvl Lib "MCUG3.DLL" (ByRef value As Double) As Integer ' read trajectory target velocity

   Declare Function wrVBError Lib "MCUG3.DLL" (an As Integer, ByRef value As Double) As Integer   ' write Velocity-Blending-Error
   Declare Function rdVBError Lib "MCUG3.DLL" (an As Integer, ByRef value As Double) As Integer   ' read Velocity-Blending-Error

   Declare Sub wrGCR Lib "MCUG3.DLL" (an As Integer, ByRef value As Integer) 'write GCR to axis
   Declare Sub rdGCR Lib "MCUG3.DLL" (an As Integer, ByRef value As Integer) 'read GCR from axis

   Declare Sub rdrpsync Lib "MCUG3.DLL" (ByRef DTSRP As TSRP)    'read real position + sync aux (only for special functionality)
   Declare Sub rdauxsync Lib "MCUG3.DLL" (ByRef DTSRP As TSRP)   'read auxiliary value syncronized (only for special functionality)
   Declare Function rdOsVersion Lib "MCUG3.DLL" (ByRef value As Integer) As Integer    ' read Version Number of Operating system rwmos.elf
   Declare Function wrSampleTimeExpansion Lib "MCUG3.DLL" (ByRef value As Integer) As Integer    ' write system var SampleTimeExpansion
   Declare Function rdSampleTimeExpansion Lib "MCUG3.DLL" (ByRef value As Integer) As Integer    ' read system var SampleTimeExpansion

   Declare Function getEnvStr Lib "MCUG3.DLL" (data() As Byte) As Integer
   Declare Sub wrControllerFlags Lib "MCUG3.DLL" (an As Integer, ByRef value As Integer) 'write Controller-Flags-Register to axis
   Declare Sub rdControllerFlags Lib "MCUG3.DLL" (an As Integer, ByRef value As Integer) 'read Controller-Flags-Register from axis
   Declare Function CleanServiceUsers Lib "MCUG3.DLL" () As Integer                   ' Clear Servic Users in rwPhysMemMgnt
End Module