/*

  +-----------------------------------------------------------------------+
  | (C) Roesch & Walter Industrie-Elektronik GmbH                         |
  | Woertelweg 2 b/c             D-77963 Schwanau                         |
  +-------------------------------+---------------------------------------+
  | Tel : +49 (0) 7824/6628-0     | email    : support@rw-gmbh.de         |
  | Fax : +49 (0) 7824/6628-29    | Internet : http://www.rw-gmbh.de      |
  +-------------------------------+---------------------------------------+
  | Project     : MCUG3 TOOLSET   | Compiler : BORLAND C++ Builder        |
  |                               |            MICROSOFT Visual C++       |
  | Module name : mcug3.h         | Operating System:                     |
  |                               |            - Windows 95/98/Me         |
  |                               |            - Windows NT 4.0           |
  |                               |            - Windows NT Embedded 4.0  |
  |                               |            - Windows 2000             |
  |                               |            - Windows XP               |
  |                               |            - Windows Vista            |
  |                               |            - Windows 7.0              |
  |                               |            - Windows 8.0              |
  +-------------------------------+---------------------------------------+
  | Author : Roesch R.               Date :12/01/2015                     |
  |          Walter M.                                                    |
  +-----------------------------------------------------------------------+
  | Description   : Driver Interface for                                  |
  |                 High-Performance Motion-Control Family Generation 3   |
  |                                                                       |
  |                 Currently supported G3-Controllers:                   |
  |                         MCU-3000 and APCI-8001                        |
  |                         MCU-6000 and APCI-8401                        |
  |                         MCU-3100 and APCI-8008                        |
  |                         MCU-3106 and APCI-8008-STP-EVAI               |
  |                                                                       |
  |                 This file contains all prototypes, definitions and    |
  |                 structures to build a Win32-based application program |
  |                                                                       |
  +-----------------------------------------------------------------------+

    Revision History:
	-----------------
	25.01.2024 	V2-53-138 mw new function ec_chk_option
	03.08.2021	V2-53-133 mw new functions wrFilename, wrFile, FileToRead, ReqSendFileBlock and rdDmesg
    04.08.2020  V2-53-130 mw new function ClearCI added
	30/04/2019	V2-53-121 mw new function CheckRwmosBusy (only for additional diag.)
	09/05/2018	V2-53-115 mw new functions rdPcapIndex, VelToAOut, SelectTable
	25/07/2016	V2-53-113 mw new functions: sstvl, spda, spdr
	17/03/2016	V2-53-111 mw new functions: rdShortProfile, wrShortProfile
							 threshold Profile length for NoTriangle (default 5.0)
	18/02/2016  V2-51-110 mw new functions: rdVBError, wrVBError for Velocity-Blending
	22/04/2015 	V2-53-108 mw new function: wrtrac, wrtrvl, wrtrtvl, rdtrac, rdtrvl, rdtrtvl
	25/02/2015 	V2-53-106 mw functions wrOptionBuf, rdOptionBuf added
    15/01/2015  V2-53-105 rr imported from mcug3.h
    before      see rev. history in mcug3.h

*/


#ifndef mcug3H
#define mcug3H

/*
  +-----------------------------------------------------------------------+
  | Language specific macro definitions ...                               |
  +-----------------------------------------------------------------------+
*/
#ifndef dllfuncH
    #ifdef __cplusplus
        #define DLLFUNCW32  extern "C"
        #define DLLFUNC     _stdcall
    #else
        #define DLLFUNCW32
        #define DLLFUNC     _stdcall
    #endif


#endif
#if defined(__BORLANDC__) || (_MSC_VER >= 800) || defined (_CVI_) || defined(_STDCALL_SUPPORTED)
    #define MCUG3CALLBACK __stdcall
#endif


/*
  +-----------------------------------------------------------------------+
  | Definitions, structures, unions and enumerations ...                  |
  +-----------------------------------------------------------------------+
*/

#define MAXAXIS         18      /* currently maximum supplied number of axis, do not change !!! */
#define SIZE_STRREV     40      /* String-Size definition */
#define SIZE_CIBUF      1000    /* Number of elements in common integer buffer */
#define SIZE_CDBUF      1000    /* Number of elements in common double buffer */
#define BSIZE           1024    /* Size of System File Header and CNC File Header */
#define SIZE_FILENAMES  256     /* Max. (char) Size for Filenames */
#define SIZE_TIME       32      /* Size (char) of System time information */
#define SCPSZ_R         15      /* row size for CTRLRPARAMS */
#define SCPSZ_C         15      /* colomn size for CTRLRPARAMS*/
#define SPD2D_S         10      /* size of SPD2D datas */

/* Enumeration for Position units */
typedef enum    {MM, INCH, M, REV, DEG,
                 RAD, COUNTS, STEPS, LAST_PU   } PUS;
/* Enumeration for Time units */
typedef enum    {SEC, MIN, TSAMPLE, LAST_TU    } TUS;

/* Enumeration for MCU-G3 Board Types */
typedef enum
{
    UNKNOWNBOARD = 0x0,         /* unknown board type */
    MCU3000      = 0x4,         /* G3-Controller MCU-3000 */
    APCI8001     = 0x4,         /* G3-Controller APCI-8001 */
    MCU6000      = 0x8,         /* G3-Controller MCU-6000 */
    APCI8401     = 0x8,         /* G3-Controller APCI-8401 */
    MCU3400      = 0x10,        /* G3-Controller MCU-3400 (Compact PCI) */
    CPCI8001     = 0x10,        /* G3-Controller CPCI8001 */
    MCU3100      = 0x20,        /* G3-Controller MCU-3100 */
    APCI8008     = 0x20         /* G3-Controller APCI-8008 */
} BOARDTYPE;

#pragma pack(4)
union AXST
{
    int status_word;
    struct {
        /* axst: Error- & Status-Bits */
        unsigned toasm  : 1;    /* Bit 0:  MCU-6000, APCI-8401 only, MCU-3000 / APCI-8001 = 0 */
        unsigned eo     : 1;    /* Bit 1:  Emergency out */
        unsigned dnr    : 1;    /* Bit 2:  Drive not ready */
        unsigned lslh   : 1;    /* Bit 3:  Limit switch left hardware */
        unsigned lsrh   : 1;    /* Bit 4:  Limit switch right hardware */
        unsigned lsls   : 1;    /* Bit 5:  Limit switch left software */
        unsigned lsrs   : 1;    /* Bit 6:  Limit switch right software */
        unsigned mpe    : 1;    /* Bit 7:  Maximum Position Error */
        unsigned dhef   : 1;    /* Bit 8:  data handling error flag */
        unsigned cef    : 1;    /* Bit 9:  configuration error flag */
        unsigned nc     : 2;    /* Bit 10..11: currently not used */

        /* Status-Bits */
        unsigned pe     : 1;    /* Bit 12: Profile end */
        unsigned cl     : 1;    /* Bit 13: Closed loop */
        unsigned ip     : 1;    /* Bit 14: In Position */
        unsigned ui     : 1;    /* Bit 15: user input */
        unsigned lpsf   : 1;    /* Bit 16: latched position synchronous flag */
        unsigned ref    : 1;    /* Bit 17: Axis is referenced (per SHP) */
        unsigned refhw  : 1;    /* Bit 18: REF HW Input is active */
        unsigned saf    : 1;    /* Bit 19: Spooler asyncronous flag */
        unsigned temp2  : 10;   /* Bit 20..29: currently not used */
        
        unsigned        : 2;    /* Bit 30..31: for internal use only */
    } bit;
};

struct TSRP
{
    /* Common-Data for DLL-interface-driver MCUG3.DLL and PC-Application-Program(s) */
    int     an;             /* Axis-Number */
    double      kp;             /* PIDF filter parameter kp */
    double      ki;             /* PIDF filter parameter ki */
    double      kd;             /* PIDF filter parameter kd */
    double      kpl;            /* PIDF filter parameter kpl */
    double      kfca;           /* PIDF forward compensation acceleration */
    double      kfcv;           /* PIDF forward compensation velocity */
    double      jac;            /* jog acceleration */
    double      jvl;            /* jog velocity */
    double      jtvl;           /* target velocity */
    double      jovr;           /* jog override */
    double      hac;            /* home acceleration */
    double      hvl;            /* home velocity */
    double      rp;             /* real position */
    double      dp;             /* desired position */
    double      tp;             /* target position */
    double      dv;             /* desired velocity */
    double      lp;             /* latched postion */
    double      aux;            /* auxilary value */
    double      sll;            /* software limit left */
    double      slr;            /* software limit right */
    double      ipw;            /* in position window */
    double      mpe;            /* maximum position error */
    double      sdec;           /* stop deceleration */
    double      gf;             /* gear factor */
    int         mcp;            /* motor command port */
    union AXST  axst;           /* axis status  */
    int         lsm;            /* left spool memory */
    
    int         epc;            /* eeprom programming cycle */
    int         digi;           /* dig. inputs = asmi (MCU-6000, APCI-8401) */
    int         digo;           /* dig outputs = asmo (MCU-6000, APCI-8401)*/
    int         ifs;            /* interface status = asms (MCU-6000, APCI-8401) */
        
    int         scratch[4];     /* reserved space don't delete!! */
};

struct AS {
    /* Axis-Selection */
    int         unoa;           /* Number of used axis */
    int         san[MAXAXIS];   /* Axis Numbers */
};                              
    
struct TRU {                    
    /* Trajectory Units */      
    int         pu;             /* position unit, see enumeration PUS above */
    int         tu;             /* time unit, see enumeration TUS above */
};

struct LMP {
    /* Linear Motion Profile parameters */
    double      ac;             /* trajectory acceleration  */
    double      vl;             /* trajectory velocity */
    double      tvl;            /* trajectory target velocity */
    double      dtm[MAXAXIS];   /* distance to move */
};

struct CMP {
    /* Circular Motion Profile parameters */
    double      ac;             /* trajectory acceleration  */
    double      vl;             /* trajectory velocity */
    double      tvl;            /* trajectory target velocity */
    double      phi;            /* angel phi */
    double      dtca1;          /* center point of axis 1 */
    double      dtca2;          /* center point of axis 2 */
};

struct HMP {
    /* Helical Motion Profile parameters */
    double      ac;             /* trajectory acceleration  */
    double      vl;             /* trajectory velocity */
    double      tvl;            /* trajectory target velocity */
    double      phi;            /* angel phi */
    double      dtca1;          /* center point of axis 1 */
    double      dtca2;          /* center point of axis 2 */
    double      dtm[MAXAXIS];   /* distance to move axis 3..n */
};

struct HMP3D {
    /* 3D Circular and Helical Motion Profile parameters */
    double      ac;         /* trajectory acceleration  */
    double      vl;         /* trajectory velocity */
    double      tvl;            /* trajectory target velocity */
    double      phi;            /* angel phi */
    double      dtca1;          /* center point of axis 1 */
    double      dtca2;          /* center point of axis 2 */
    double      dtca3;          /* center point of axis 3 */
    double      pn1;            /* area perpendicular of axis 1 (Flchen-Normale A1) */
    double      pn2;            /* area perpendicular of axis 2 (Flchen-Normale A2) */
    double      pn3;            /* area perpendicular of axis 3 (Flchen-Normale A3) */
    double      dtma[MAXAXIS];          /* distance to move axis 3..n */
};

struct CNCTS {
    /* CNC-Task Status variables */
    int         ErrNum;         /* error number */
    int         ErrLine;        /* error line */
    int         StackFree;      /* stack bytes free */
    int         Running;        /* nc task running status */
    int         csrcline;       /* current (processed) source line number */
};

struct ROSI
{
    /* RISC Operating System Informations */
    char        revision[SIZE_STRREV];  
                                /* software-revision-string */
    int         Number_Axis;    /* number of supplied axis */
    int         SysFile_Loaded; /* status flag for sysfile already loaded */
};

struct CIBUF
{
    /* Buffer definition for common integer variables */
    int         civals[SIZE_CIBUF]; 
};

struct CDBUF
{
    /* Buffer definition for common double variables */
    double      cdvals[SIZE_CDBUF]; 
};

struct CBCNCT
{
    /* Common CNC-Task buffer */
    int         TaskNr;         /* Task Number 0..3 */
    int         size;           /* size of Buffer [bytes] */
    char        *Buffer;        /* pointer to user allocated buffer */
};

struct SPDBUF
{
    /* position data to spool */
    double      dtm;            /* distance to move */
};

struct SPVTDBUF
{
    /* position, velocity, time datas to spool */
    double      dtm;            /* distance to move */
    double      vl;             /* velocity */
    double      t;              /* time */
};

struct CTRLRPARAMS {
    /* Controller specific data fields (application specicfic) */
    double      val[SCPSZ_R][SCPSZ_C];
};

struct SPD2DPARAMS
{
    int         size;           /* size */
    int         san[2];         /* selected axis numbers */
    double      dtma1[SPD2D_S]; /* distance(s) to move axis 1 */
    double      dtma2[SPD2D_S]; /* ditsance(s) to move axis 2 */
};

struct LATCH_INFOS
{
    /* parameter set for latching position datas */
    int         san;            /* selected axis number */
    double      lst;            /* latch start time (e.g. 0) */
    double      let;            /* latch end time (e.g. 1) */
    int         pu;             /* position unit (see enum PUS) */
    int         tu;             /* time unit (see enum TUS) */
};

/* Datenstruktur zum einlesen der Grafikanalyse */
struct GRFFILEDATA {
    char Header[256];
    int  EndOfFile;
    struct LATCH_INFOS  li;

    double RPBuffer[1000];
    double DPBuffer[1000];
};

struct OLSR_INFOS {
    /* parameter set for open loop system response */
    int         san;            /* selected axis number */
    double      delay;          /* delay [s] */
    double      mcp;            /* motor command value [V] */
};

struct FILEHEADER
{
    /* CNC Task File Header Information */
    int         TaskNr;         /* NC-File runs in task 'TaskNr' 0..3 */
    char        SrcFileName[SIZE_FILENAMES];    
                                /* CNC Source File Name */
    char        CncFileName[SIZE_FILENAMES];    
                                /* CNC pseudo code file name */
    char        RevNr[SIZE_STRREV];             
                                /* revision number */
    char        DateTime[SIZE_TIME];            
                                /* build date and time information */
};

struct OptionDescriptorObject
{
    unsigned    Handle;         /* do not change that! */
    unsigned    AccessType;     /* Read, Write, Flag */
    unsigned    DataType;       /* Byte, Word, Double Word, Real, ... */
    unsigned    BusNumber;      /* Interbus, Canbus, G3-Resourcebus, Scannerbus ... */
    unsigned    DeviceNumber;   /* depends on Bus Type */
    unsigned    Index;          /* depends on Bus Type */
    unsigned    SubIndex;       /* depends on Bus Type */
};

enum ATAccessType {ATAccessNone, ATAccessInput, ATAccessOutput, ATAccessInputOutput};
enum ATDataType   {ATDataNone,   ATDataByte,    ATDataWord,     ATDataDoubleWord, ATDataReal, ATDataSingle, ATDataBlock, ATDataBoolean, ATDataBuffer};
enum ATBusNr {
    PciBusIO        = 100,
    PciBusMEM       = 200,
    CanOpenBus      = 400,
    InterBus        = 500,
    ECBus           = 600,
    ResourcenBus    = 1000,
    ScannerBus      = 1100,
    ElCamBus        = 1200,
    TcBus           = 1300,
    CotBus          = 2000
};

#pragma pack()

/*
  +-----------------------------------------------------------------------+
  | Prototypes standard functions,                                        |
  | functions are described in MCU-G3 Programming Manual                  |
  +-----------------------------------------------------------------------+
*/

DLLFUNCW32 void DLLFUNC uf                  (int CardNumber, struct TSRP *tsrp);                /* update filter */
DLLFUNCW32 void DLLFUNC ctru                (int CardNumber, struct TRU  *tru);                 /* change trajectory units */
DLLFUNCW32 void DLLFUNC wrjac               (int CardNumber, struct TSRP *tsrp);                /* write jog acceleration */
DLLFUNCW32 void DLLFUNC wrjvl               (int CardNumber, struct TSRP *tsrp);                /* write jog velocity */
DLLFUNCW32 void DLLFUNC wrjtvl              (int CardNumber, struct TSRP *tsrp);                /* write jog target velocity */
DLLFUNCW32 void DLLFUNC wrhac               (int CardNumber, struct TSRP *tsrp);                /* write home acceleration */
DLLFUNCW32 void DLLFUNC wrhvl               (int CardNumber, struct TSRP *tsrp);                /* write home velocity */
DLLFUNCW32 void DLLFUNC wrrp                (int CardNumber, struct TSRP *tsrp);                /* write real position */
DLLFUNCW32 void DLLFUNC wrdp                (int CardNumber, struct TSRP *tsrp);                /* write desired position */
DLLFUNCW32 void DLLFUNC wrlp                (int CardNumber, struct TSRP *tsrp);                /* write latched position */
DLLFUNCW32 void DLLFUNC wraux               (int CardNumber, struct TSRP *tsrp);                /* write auxiliary value */
DLLFUNCW32 void DLLFUNC shp                 (int CardNumber, struct TSRP *tsrp);                /* set home position */
DLLFUNCW32 void DLLFUNC wrdigo              (int CardNumber, struct TSRP *tsrp);                /* MCU-3000, APCI-8001 only: write digital outputs */
DLLFUNCW32 void DLLFUNC wrdigob             (int CardNumber, int an, int bitnr, int value);     /* MCU-3000, APCI-8001 only: write digital output bit */
DLLFUNCW32 void DLLFUNC wrasmo              (int CardNumber, struct TSRP *tsrp);                /* MCU-6000, APCI-8401 only: write ASM digital outputs */
DLLFUNCW32 void DLLFUNC wrasmob             (int CardNumber, int an, int bitnr, int value);     /* MCU-6000, APCI-8401 only: write ASM digital output bit */
DLLFUNCW32 void DLLFUNC rdrp                (int CardNumber, struct TSRP *tsrp);                /* read real position */
DLLFUNCW32 void DLLFUNC rddp                (int CardNumber, struct TSRP *tsrp);                /* read desired position */
DLLFUNCW32 void DLLFUNC rdtp                (int CardNumber, struct TSRP *tsrp);                /* read target position */
DLLFUNCW32 void DLLFUNC rddv                (int CardNumber, struct TSRP *tsrp);                /* read desired velocity */
DLLFUNCW32 void DLLFUNC rdlp                (int CardNumber, struct TSRP *tsrp);                /* read latched position */
DLLFUNCW32 void DLLFUNC rdaux               (int CardNumber, struct TSRP *tsrp);                /* read auxiliary value */
DLLFUNCW32 void DLLFUNC rdauxf              (int CardNumber, struct TSRP *tsrp);                /* read auxiliary value fast */
DLLFUNCW32 void DLLFUNC rddigi              (int CardNumber, struct TSRP *tsrp);                /* MCU-3000, APCI-8001 only: read digital inputs */
DLLFUNCW32 int  DLLFUNC rddigib             (int CardNumber, int an, int bitnr);                /* MCU-3000, APCI-8001 only: read digital input bit */
DLLFUNCW32 int  DLLFUNC rdifs               (int CardNumber, struct TSRP *tsrp);                /* MCU-3000, APCI-8001 only: read interface status */
DLLFUNCW32 int  DLLFUNC rdifsb              (int CardNumber, int an, int bitnr);                /* MCU-3000, APCI-8001 only: read interface status bit */
DLLFUNCW32 void DLLFUNC rddigo              (int CardNumber, struct TSRP *tsrp);                /* MCU-3000, APCI-8001 only: read digital outputs */
DLLFUNCW32 void DLLFUNC rddigof             (int CardNumber, struct TSRP *tsrp);                /* MCU-3000, APCI-8001 only: read digital outputs direct: respect comment in header */
DLLFUNCW32 int  DLLFUNC rddigob             (int CardNumber, int an, int bitnr);                /* MCU-3000, APCI-8001 only: read digital output bit */
DLLFUNCW32 void DLLFUNC rdasmi              (int CardNumber, struct TSRP *tsrp);                /* MCU-6000, APCI-8401 only: read ASM digital inputs */
DLLFUNCW32 int  DLLFUNC rdasmib             (int CardNumber, int an, int bitnr);                /* MCU-6000, APCI-8401 only: read ASM digital input bit */
DLLFUNCW32 void DLLFUNC rdasms              (int CardNumber, struct TSRP *tsrp);                /* MCU-6000, APCI-8401 only: read ASM status */
DLLFUNCW32 int  DLLFUNC rdasmsb             (int CardNumber, int an, int bitnr);                /* MCU-6000, APCI-8401 only: read ASM status bit */
DLLFUNCW32 void DLLFUNC rdepc               (int CardNumber, struct TSRP *tsrp);                /* MCU-6000, APCI-8401 only: read eeprom programming cycle */
DLLFUNCW32 void DLLFUNC rdasmepc            (int CardNumber, struct TSRP *tsrp);                /* MCU-6000, APCI-8401 only: read ASM eeprom programming cycle */
DLLFUNCW32 void DLLFUNC rdasmo              (int CardNumber, struct TSRP *tsrp);                /* MCU-6000, APCI-8401 only: read ASM digital outputs */
DLLFUNCW32 int  DLLFUNC rdasmob             (int CardNumber, int an, int bitnr);                /* MCU-6000, APCI-8401 only: read ASM digital output bit */
DLLFUNCW32 void DLLFUNC rdlsm               (int CardNumber, struct TSRP *tsrp);                /* read left spool memory */
DLLFUNCW32 void DLLFUNC rdaxst              (int CardNumber, struct TSRP *tsrp);                /* read axis status */
DLLFUNCW32 void DLLFUNC rdaxstf             (int CardNumber, struct TSRP *tsrp);                /* read axis status fast */
DLLFUNCW32 int  DLLFUNC rdaxstb             (int CardNumber, int an, int bitnr);                /* read axis status bit */
DLLFUNCW32 void DLLFUNC rdjac               (int CardNumber, struct TSRP *tsrp);                /* read jog acceleration */
DLLFUNCW32 void DLLFUNC rdjvl               (int CardNumber, struct TSRP *tsrp);                /* read jog velocity */
DLLFUNCW32 void DLLFUNC rdjtvl              (int CardNumber, struct TSRP *tsrp);                /* read jog target velocity */
DLLFUNCW32 void DLLFUNC rdhac               (int CardNumber, struct TSRP *tsrp);                /* read home acceleration */
DLLFUNCW32 void DLLFUNC rdhvl               (int CardNumber, struct TSRP *tsrp);                /* read home velocity */
DLLFUNCW32 void DLLFUNC cl                  (int CardNumber, struct AS *as);                    /* close loop */
DLLFUNCW32 void DLLFUNC clv                 (int CardNumber, struct AS *as);                    /* close loop velocity */
DLLFUNCW32 void DLLFUNC ol                  (int CardNumber, struct AS *as);                    /* open loop */
DLLFUNCW32 void DLLFUNC ra                  (int CardNumber, struct AS *as);                    /* reset axis */
DLLFUNCW32 void DLLFUNC rifs                (int CardNumber, struct TSRP *tsrp);                /* MCU-3000, APCI-8001 only: reset interface status flags */
DLLFUNCW32 void DLLFUNC rasms               (int CardNumber, struct TSRP *tsrp);                /* MCU-6000, APCI-8401 only: reset ASM status flags */
DLLFUNCW32 void DLLFUNC rs                  (int CardNumber);                                   /* reset system */
DLLFUNCW32 void DLLFUNC azo                 (int CardNumber, int set);                          /* activate zero offset */
DLLFUNCW32 void DLLFUNC jr                  (int CardNumber, struct AS *as, struct TSRP *tsrp); /* jog relative */
DLLFUNCW32 void DLLFUNC ja                  (int CardNumber, struct AS *as, struct TSRP *tsrp); /* jog absolute */
DLLFUNCW32 void DLLFUNC jhl                 (int CardNumber, struct AS *as);                    /* jog home left */
DLLFUNCW32 void DLLFUNC jhr                 (int CardNumber, struct AS *as);                    /* jog home right */
DLLFUNCW32 void DLLFUNC jhi                 (int CardNumber, struct AS *as, struct TSRP *tsrp); /* jog relative (to) index */
DLLFUNCW32 void DLLFUNC mlr                 (int CardNumber, struct AS *as, struct LMP *lmp);   /* move linear relative */
DLLFUNCW32 void DLLFUNC mla                 (int CardNumber, struct AS *as, struct LMP *lmp);   /* move linear absolute */
DLLFUNCW32 void DLLFUNC mcr                 (int CardNumber, struct AS *as, struct CMP *cmp);   /* move circular relative */
DLLFUNCW32 void DLLFUNC mca                 (int CardNumber, struct AS *as, struct CMP *cmp);   /* move circular absolute */
DLLFUNCW32 void DLLFUNC mhr                 (int CardNumber, struct AS *as, struct HMP *hmp);   /* move helix relative */
DLLFUNCW32 void DLLFUNC mha                 (int CardNumber, struct AS *as, struct HMP *hmp);   /* move helix absolute */
DLLFUNCW32 void DLLFUNC mca3d               (int CardNumber, struct AS *as, struct HMP3D *hmp3d); /* move 3D-circular-absolute */
DLLFUNCW32 void DLLFUNC mcr3d               (int CardNumber, struct AS *as, struct HMP3D *hmp3d); /* move 3D-circular-relative */
DLLFUNCW32 void DLLFUNC smca3d              (int CardNumber, struct AS *as, struct HMP3D *hmp3d); /* move spooled 3D-circular-absolute */
DLLFUNCW32 void DLLFUNC smcr3d              (int CardNumber, struct AS *as, struct HMP3D *hmp3d); /* move spooled 3D-circular-relative */

DLLFUNCW32 void DLLFUNC smlr                (int CardNumber, struct AS *as, struct LMP *lmp);   /* spool motion linear relative */
DLLFUNCW32 void DLLFUNC smlri               (int CardNumber, struct AS *as, struct LMP *lmp, int PcapIndex);
                                                                                                /* spool motion linear relative indicated */
DLLFUNCW32 void DLLFUNC smla                (int CardNumber, struct AS *as, struct LMP *lmp);   /* spool motion linear absolute */
DLLFUNCW32 void DLLFUNC smlai               (int CardNumber, struct AS *as, struct LMP *lmp, int PcapIndex);
                                                                                                /* spool motion linear absolute indicated */
DLLFUNCW32 void DLLFUNC smcr                (int CardNumber, struct AS *as, struct CMP *cmp);   /* spool motion circular relative */
DLLFUNCW32 void DLLFUNC smca                (int CardNumber, struct AS *as, struct CMP *cmp);   /* spool motion circular absolute */
DLLFUNCW32 void DLLFUNC smhr                (int CardNumber, struct AS *as, struct HMP *hmp);   /* spool motion helix relative */
DLLFUNCW32 void DLLFUNC smha                (int CardNumber, struct AS *as, struct HMP *hmp);   /* spool motion helix absolute */
DLLFUNCW32 void DLLFUNC ssms                (int CardNumber, struct AS *as);                    /* start spooled motions synchronous */
DLLFUNCW32 void DLLFUNC sstps               (int CardNumber, struct AS *as);                    /* stop spooled motions synchronous */
DLLFUNCW32 void DLLFUNC sdels               (int CardNumber, struct AS *as);                    /* delete spooled motions synchronous */
DLLFUNCW32 int  DLLFUNC txbf                (int CardNumber, char *filename);                   /* transmit binary file */
DLLFUNCW32 int  DLLFUNC txbf2               (int CardNumber, char *filename);                   /* transmit binary file, 2nd method */
DLLFUNCW32 void DLLFUNC startcnct           (int CardNumber, int TaskNr);                       /* start cnc task */
DLLFUNCW32 void DLLFUNC stopcnct            (int CardNumber, int TaskNr);                       /* stop cnc task */
DLLFUNCW32 void DLLFUNC contcnct            (int CardNumber, int TaskNr);                       /* continue cnc task */
DLLFUNCW32 void DLLFUNC stepcnct            (int CardNumber, int TaskNr);                       /* step cnc task */
DLLFUNCW32 void DLLFUNC rdcncts             (int CardNumber, int TaskNr, struct CNCTS *cncts);  /* read cnc task status */
DLLFUNCW32 int  DLLFUNC rdcbcnct            (int CardNumber, struct CBCNCT *cbcnct);            /* read cnc task spec. common buffer */
DLLFUNCW32 int  DLLFUNC wrcbcnct            (int CardNumber, struct CBCNCT *cbcnct);            /* write cnc task spec. common buffer */
DLLFUNCW32 void DLLFUNC rdci                (int CardNumber, int ndx, struct CIBUF *cibuf);     /* read common integer */
DLLFUNCW32 void DLLFUNC rdcd                (int CardNumber, int ndx, struct CDBUF *cdbuf);     /* read common double */
DLLFUNCW32 void DLLFUNC wrci                (int CardNumber, int ndx, struct CIBUF *cibuf);     /* write common integer variables */
DLLFUNCW32 void DLLFUNC wrci_               (int CardNumber, int ndx, struct CIBUF *cibuf);     /* write common integer variables per Communication (slower) */
DLLFUNCW32 void DLLFUNC wrcd                (int CardNumber, int ndx, struct CDBUF *cdbuf);     /* write common double variables */
DLLFUNCW32 void DLLFUNC rdmcp               (int CardNumber, struct TSRP *tsrp);                /* read motor command port */
DLLFUNCW32 void DLLFUNC wrmcp               (int CardNumber, struct TSRP *tsrp);                /* write motor command port */
DLLFUNCW32 void DLLFUNC wrledrd             (int CardNumber, int value);                        /* write led red */
DLLFUNCW32 void DLLFUNC wrledyl             (int CardNumber, int value);                        /* write led yellow */
DLLFUNCW32 void DLLFUNC wrledgn             (int CardNumber, int value);                        /* write led green */
DLLFUNCW32 int  DLLFUNC rdledrd             (int CardNumber);                                   /* read led red */
DLLFUNCW32 int  DLLFUNC rdledyl             (int CardNumber);                                   /* read led yellow */
DLLFUNCW32 int  DLLFUNC rdledgn             (int CardNumber);                                   /* read led green */
DLLFUNCW32 int  DLLFUNC rdirqpc             (int CardNumber);                                   /* read irq pc */
DLLFUNCW32 int  DLLFUNC mcuinit             (int CardNumber, struct ROSI *rosi);                /* MCU-G3 initialisition */
DLLFUNCW32 int  DLLFUNC mcuinit2            (int CardNumber, struct ROSI *rosi, int TpuBaseAddress);
                                                                                                /* MCU-G3 initialisition, 2nd method */
DLLFUNCW32 void DLLFUNC wrjovr              (int CardNumber, struct TSRP *tsrp);                /* write jog override */
DLLFUNCW32 void DLLFUNC rdjovr              (int CardNumber, struct TSRP *tsrp);                /* read jog override */
DLLFUNCW32 void DLLFUNC wrtrovr             (int CardNumber, double *value);                    /* write trajectory override */
DLLFUNCW32 void DLLFUNC rdtrovr             (int CardNumber, double *value);                    /* read trajectory override */
DLLFUNCW32 int  DLLFUNC wrtrovrst           (int CardNumber, double *value);                    /* write trajectory override settling time*/
DLLFUNCW32 int  DLLFUNC rdtrovrst           (int CardNumber, double *value);                    /* read trajectory override settling time */
DLLFUNCW32 int  DLLFUNC wrShortProfile      (int CardNumber, double *value);                    /* write System Var ShortProfile - threshold profile length for NoTriangle */
DLLFUNCW32 int  DLLFUNC rdShortProfile      (int CardNumber, double *value);                    /* read System Var ShortProfile - threshold profile length for NoTriangle */
DLLFUNCW32 void DLLFUNC utrovr              (int CardNumber, struct AS *as);                    /* update trajectory override */
DLLFUNCW32 void DLLFUNC rdf                 (int CardNumber, struct TSRP *tsrp);                /* read filter */
DLLFUNCW32 void DLLFUNC rdap                (int CardNumber, struct TSRP *tsrp);                /* read axis parameters */
DLLFUNCW32 void DLLFUNC rdsll               (int CardNumber, struct TSRP *tsrp);                /* read software limit left */
DLLFUNCW32 void DLLFUNC wrsll               (int CardNumber, struct TSRP *tsrp);                /* write software limit left */
DLLFUNCW32 void DLLFUNC rdslr               (int CardNumber, struct TSRP *tsrp);                /* read software limit right */
DLLFUNCW32 void DLLFUNC wrslr               (int CardNumber, struct TSRP *tsrp);                /* write software limit right */
DLLFUNCW32 void DLLFUNC rdipw               (int CardNumber, struct TSRP *tsrp);                /* read in position window */
DLLFUNCW32 void DLLFUNC wripw               (int CardNumber, struct TSRP *tsrp);                /* write in position window */
DLLFUNCW32 void DLLFUNC rdmpe               (int CardNumber, struct TSRP *tsrp);                /* read maximum position error */
DLLFUNCW32 void DLLFUNC wrmpe               (int CardNumber, struct TSRP *tsrp);                /* write maximum position error */
DLLFUNCW32 void DLLFUNC rdsdec              (int CardNumber, struct TSRP *tsrp);                /* read stop deceleration */
DLLFUNCW32 void DLLFUNC wrsdec              (int CardNumber, struct TSRP *tsrp);                /* write stop deceleration */
DLLFUNCW32 void DLLFUNC spd                 (int CardNumber, int an, int size, struct SPDBUF *spdbuf);						/* spool position data */
DLLFUNCW32 void DLLFUNC spda				(int CardNumber, int an, int size, struct SPDBUF *spdbuf, long PcapIndex);		/* spool position data absolute indicated */
DLLFUNCW32 void DLLFUNC spdr				(int CardNumber, int an, int size, struct SPDBUF *spdbuf, long PcapIndex);     	/* spool position data relative indicated */
DLLFUNCW32 void DLLFUNC sstvl                (int CardNumber, struct AS *as, double tvl);        /* spooler set target velocity */
																								/* spool position data */
DLLFUNCW32 void DLLFUNC wrgf                (int CardNumber, struct TSRP *tsrp);                /* write gear factor */
DLLFUNCW32 void DLLFUNC rdgf                (int CardNumber, struct TSRP *tsrp);                /* read gear factor */
DLLFUNCW32 void DLLFUNC spvtd               (int CardNumber, int an, int size, struct SPVTDBUF *spvtdbuf);
                                                                                                /* spool position, veloctiy, time data */
DLLFUNCW32 void DLLFUNC ms                  (int CardNumber, struct AS *as);                    /* motion stop */
DLLFUNCW32 void DLLFUNC js                  (int CardNumber, struct AS *as);                    /* jog stop */
DLLFUNCW32 void DLLFUNC lps                 (int CardNumber, int an, int mst);                  /* latch position synchronous */
DLLFUNCW32 void DLLFUNC rdigi               (int CardNumber, struct TSRP *tsrp);                /* MCU-3000, APCI-8001 only: reset digital inputs */
DLLFUNCW32 void DLLFUNC rdlpndx             (int CardNumber, struct TSRP *tsrp);                /* MCU-3000, APCI-8001 only: read latched position index */
DLLFUNCW32 void DLLFUNC wrlpndx             (int CardNumber, struct TSRP *tsrp);                /* MCU-3000, APCI-8001 only: write latched position index */
DLLFUNCW32 void DLLFUNC ssf                 (int CardNumber, int an, int command, double value);/* Spooler-Special-Function-Command */
DLLFUNCW32 void DLLFUNC ssfn                (int CardNumber, struct AS *as, long command, double value);
                                                                                                /* SSF fr mehr als eine Achse */
DLLFUNCW32 void DLLFUNC ssfi                (int CardNumber, int an, int command, double value, int PcapIndex);
                                                                                                /* Spooler-Special-Function-Command indicated */
DLLFUNCW32 void DLLFUNC ssfni               (int CardNumber, struct AS *as, long command, double value, int PcapIndex);
                                                                                                /* SSF fr mehr als eine Achse indicated */

DLLFUNCW32 int  DLLFUNC InitMcuSystem       (int CardNumber, struct TSRP *tsrp);                /* initialisation function first method, (you should not use this function, instead use InitMcuSystem2() or even better InitMcuSystem3()) */
DLLFUNCW32 int  DLLFUNC InitMcuSystem2      (int CardNumber, struct TSRP *tsrp, int TpuBaseAddress, char *SystemFileName);
                                                                                                /* initialisation function, 2nd method*/
DLLFUNCW32 int  DLLFUNC InitMcuSystem3      (int CardNumber, struct TSRP *tsrp, struct ROSI *rosi, int TpuBaseAddress, char *SystemFileName, int *BoardType);
                                                                                                /* initialisation function, 3rd method*/
DLLFUNCW32 int  DLLFUNC InitMcuSystem4      (int CardNumber, struct TSRP *tsrp, struct ROSI *rosi, int TpuBaseAddress, char *SystemFileName, int *BoardType, int *NrCards);
                                                                                                /* initialisation function, 4th method (method 3 with additional number of cards result) */
DLLFUNCW32 void DLLFUNC InitMcuErrorReport  (int CardNumber, int error);                        /* detailed initialisation error report function */
DLLFUNCW32 void DLLFUNC txbfErrorReport     (int CardNumber, char *filename, int error);        /* detailed load error report function */
DLLFUNCW32 void DLLFUNC BootErrorReport     (int CardNumber, char *filename, int error);        /* detailed boot error report function */
DLLFUNCW32 void DLLFUNC McuClose            (int CardNumber);                                   /* close driver function, must be called at end of program !*/
DLLFUNCW32 int  DLLFUNC BootFile            (int CardNumber, char *BootFileName, int TpuBaseAddress);
                                                                                                /* Send Bootfile (i.e. rwmos.elf) to controller board */
DLLFUNCW32 void DLLFUNC LdSSplineA          (int CardNumber, struct AS *as, struct LMP *lmp);   /* Load Spool Spline Data Absolut */
DLLFUNCW32 void DLLFUNC LdSSplineR          (int CardNumber, struct AS *as, struct LMP *lmp);   /* Load Spool Spline Data Relativ */
DLLFUNCW32 void DLLFUNC ExeSSpline          (int CardNumber, struct AS *as, struct LMP *lmp);   /* Execute Spool Spline */

DLLFUNCW32 int  DLLFUNC wrOptionInt         (int CardNumber, struct OptionDescriptorObject *odesc, int *val);
                                                                                                /* Option Object write integer function */
DLLFUNCW32 int  DLLFUNC rdOptionInt         (int CardNumber, struct OptionDescriptorObject *odesc, int *val);
                                                                                                /* Option Object read integer function */
DLLFUNCW32 int  DLLFUNC wrOptionDbl         (int CardNumber, struct OptionDescriptorObject *odesc, double *val);
																								/* Option Object write double function */
DLLFUNCW32 int  DLLFUNC rdOptionDbl         (int CardNumber, struct OptionDescriptorObject *odesc, double *val);
                                                                                                /* Option Object read double function */
DLLFUNCW32 int  DLLFUNC wrOptionBuf         (int CardNumber, struct OptionDescriptorObject *odesc, void *val);
																				/* Option Object write Buffer function */
DLLFUNCW32 int  DLLFUNC rdOptionBuf         (int CardNumber, struct OptionDescriptorObject *odesc, void *val);
																				/* Option Object read Buffer function */
DLLFUNCW32 int  DLLFUNC freeOptionHandle    (int CardNumber, struct OptionDescriptorObject *odesc);
                                                                                                /* Option Object free handle */

DLLFUNCW32 int  DLLFUNC rdScannerBufferSize (int CardNumber);                                   /* Read size of scan buffer */
DLLFUNCW32 int  DLLFUNC rdScannerStatus     (int CardNumber);                                   /* read scanner status register */
DLLFUNCW32 int  DLLFUNC rdScannerLsm        (int CardNumber);                                   /* read scanner left spool memory */
DLLFUNCW32 int  DLLFUNC rdScannerBuffer     (int CardNumber, char *buffer, int size);           /* read scanner buffer */
DLLFUNCW32 int  DLLFUNC SendReqScannerBuffer (int CardNumber, char *buffer, int size);          /* read scanner buffer */
DLLFUNCW32 void DLLFUNC CleanServiceData 	(void);												/* freeing PhysMem in rwPhysMemService (optional) */

DLLFUNCW32 void DLLFUNC wrModeReg(int CardNumber, long *value);                                 /* write ModeRegister */
DLLFUNCW32 void DLLFUNC rdModeReg(int CardNumber, long *value);                                 /* read  ModeRegister */
DLLFUNCW32 void DLLFUNC wrnfrax  (int CardNumber, long *value);                                 /* write No-Feed-Rate-Axis */
DLLFUNCW32 void DLLFUNC rdnfrax  (int CardNumber, long *value);                                 /* read  No-Feed-Rate-Axis */
DLLFUNCW32 void DLLFUNC wrMDVel(int CardNumber, long an, double *value);                        /* write Maximum Delta Velocity */
DLLFUNCW32 void DLLFUNC rdMDVel(int CardNumber, long an, double *value);                        /* read  Maximum Delta Velocity */
DLLFUNCW32 void DLLFUNC wrMaxVel(int CardNumber, long an, double *value);                       /* write Maximum Axis Velocity */
DLLFUNCW32 void DLLFUNC rdMaxVel(int CardNumber, long an, double *value);                       /* read  Maximum Axis Velocity */
DLLFUNCW32 void DLLFUNC wrMaxAcc(int CardNumber, long an, double *value);                       /* write Maximum Axis Acceleration */
DLLFUNCW32 void DLLFUNC rdMaxAcc(int CardNumber, long an, double *value);                       /* read  Maximum Axis Acceleration */
DLLFUNCW32 void DLLFUNC wrEffRadius(int CardNumber, long an, double *value);                    /* write Effective Radius for rotatoric Axis */
DLLFUNCW32 void DLLFUNC rdEffRadius(int CardNumber, long an, double *value);                    /* read  Effective Radius for rotatoric Axis */
DLLFUNCW32 void DLLFUNC rdPosErr(int CardNumber, long an, double *value);                       /* read  Position Error */
DLLFUNCW32 void DLLFUNC rdMCiS(int CardNumber, long an, long *value);                           /* read  Move-Commands-in-Spooler */
DLLFUNCW32 int  DLLFUNC rdZeroOffset(int CardNumber, int an, double *value);                    /* read  absolute Value of actual set Zero Offset */
DLLFUNCW32 int  DLLFUNC rdptp(int CardNumber, int an, double *value);                           /* read  profile target position */
DLLFUNCW32 int  DLLFUNC rdpir(int CardNumber, int *value);                                      /* read  Profile Information Register */
DLLFUNCW32 int  DLLFUNC rddtv(int CardNumber, double *value);                                   /* read  desired trajectory velocity */
DLLFUNCW32 int  DLLFUNC rdSampleTimeExpansion(int CardNumber, int *value);                      /* read  SampleTimeExpansion Register */
DLLFUNCW32 int  DLLFUNC wrSampleTimeExpansion(int CardNumber, int *value);                      /* write  SampleTimeExpansion Register */
DLLFUNCW32 void DLLFUNC RPtoDP (int CardNumber, struct AS *as);                                 /* Real Position To Desired Position */
DLLFUNCW32 int  DLLFUNC rdpeaux (int CardNumber, int an, double *value);                        /* read difference rp - aux in [UU] */
DLLFUNCW32 int  DLLFUNC wrVBError(int CardNumber, int an, double *value);                       /* write Velocity-Blending-Error */
DLLFUNCW32 int  DLLFUNC rdVBError(int CardNumber, int an, double *value);                       /* read  Velocity-Blending-Error */

/*
  +---------------------------------------------------------------------------------+
  |    Prototypes for client and application specific functions,                    |
  |    do not use that functions if you have no information                         |
  |    about the individual functions !!!                                           |
  |    (normaly you should not need that functions)                                 |
  +---------------------------------------------------------------------------------+
*/

DLLFUNCW32 void DLLFUNC wrBlk               (int CardNumber, char *buffer);                     /* Common Write Command without return value (internal function) */
DLLFUNCW32 int  DLLFUNC rdBlk               (int CardNumber, char *buffer);                     /* Common Read Command */
DLLFUNCW32 void DLLFUNC wrtp                (int CardNumber, struct TSRP *tsrp);                /* write target position */
DLLFUNCW32 void DLLFUNC rdrpd               (int CardNumber, struct TSRP *tsrp);                /* read real position display */
DLLFUNCW32 void DLLFUNC rddpd               (int CardNumber, struct TSRP *tsrp);                /* read desired position display */
DLLFUNCW32 void DLLFUNC rdtpd               (int CardNumber, struct TSRP *tsrp);                /* read target position display */
DLLFUNCW32 void DLLFUNC scp                 (int CardNumber, int an, struct CTRLRPARAMS *ctrlrparams);
                                                                                /* set controller parameters */
DLLFUNCW32 void DLLFUNC rdrv                (int CardNumber, int an, double *value);            /* read actual velocity */
DLLFUNCW32 void DLLFUNC cm                  (int CardNumber, int an, int *value);               /* change mode */
DLLFUNCW32 void DLLFUNC spd2d               (int CardNumber, struct SPD2DPARAMS *);             /* spool position datas 2D */
DLLFUNCW32 void DLLFUNC rdlrpr              (int CardNumber, struct LATCH_INFOS *latch_infos, double *buffer);
                                                                                                /* read latched real pos. registers */
DLLFUNCW32 void DLLFUNC rdldpr              (int CardNumber, struct LATCH_INFOS *latch_infos, double *buffer);
                                                                                                /* read latched desired pos. registers */
DLLFUNCW32 void DLLFUNC rdlmcpr             (int CardNumber, struct LATCH_INFOS *latch_infos, double *buffer);
                                                                                                /* read latched mcp registers */
DLLFUNCW32 void DLLFUNC getcnctsrclinenr    (int CardNumber, int TaskNr, int *LineNr);          /* read in current linenumber beeing processed of selected cnc task */
DLLFUNCW32 int  DLLFUNC getmsgstr           (int CardNumber, int an, char *result);             /* read in internal MCU-G3 debug-string */
DLLFUNCW32 int  DLLFUNC gettskstr           (int CardNumber, int TaskNr, char *buffer, int szbuffer);
                                                                                                /* read in internal MCU-G3 Task-string */
DLLFUNCW32 void DLLFUNC getcnctfh           (int CardNumber, int TaskNr, struct FILEHEADER *fh);
                                                                                                /* read in CNC-task-fileheader of selected cnc task */
DLLFUNCW32 void DLLFUNC burnaplpar          (int CardNumber, int an);                           /* burn appliance parameters */
DLLFUNCW32 void DLLFUNC gettimers           (int CardNumber, struct TSRP *tsrp, int an);        /* read in several debug-timers of high-prior task */
DLLFUNCW32 void DLLFUNC lpr                 (int CardNumber, struct LATCH_INFOS *latch_infos);  /* latch position registers */
DLLFUNCW32 void DLLFUNC lprs                (int CardNumber, struct AS *as, struct LATCH_INFOS *latch_infos);
                                                                                                /* latch position registers synchronous on several axis */
DLLFUNCW32 void DLLFUNC olsr                (int CardNumber, struct OLSR_INFOS *olsr);          /* open loop system response */
DLLFUNCW32 void DLLFUNC rdasmecs            (int CardNumber, struct TSRP *tsrp);                /* MCU-6000, APCI-8401 only: read in ASM eprom checksum of firmware */
DLLFUNCW32 void DLLFUNC ReadLinkStatus      (int CardNumber, int *fae, int *ba);                /* read current link stati */
DLLFUNCW32 int  DLLFUNC rdAin               (int CardNumber, int an, int channel);              /* MCU-3000, APCI-8001 only: read Analog Inputs from OPMF-3001 */
DLLFUNCW32 void DLLFUNC szpa                (int CardNumber, int an, double Position);          /* Set Zero Position absolut (virtual zero Point) */
DLLFUNCW32 void DLLFUNC szpr                (int CardNumber, int an, double Position);          /* Set Zero Position relativ (virtual zero Point) */

DLLFUNCW32 void DLLFUNC rdMcuStatus(int CardNumber, char *buffer, int size);                    /* read in MCU G3 status registers */
DLLFUNCW32 void DLLFUNC rdAsmSecureParams(int CardNumber, int an, char *buffer);                /* MCU-6000, APCI-8401 only: read ASM-EEPROM parameters (firmware data) */
DLLFUNCW32 void DLLFUNC wrAsmSecureParams(int CardNumber, int an, char *buffer);                /* MCU-6000, APCI-8401 only: write ASM-EEPROM parameters (firmware data) */


typedef void (MCUG3CALLBACK* InterruptRoutine)(int i_IRQLineBits);                              /* Callback-Prototyp for User-Interrupt-Handler */
DLLFUNCW32 int DLLFUNC MCUG3_SetBoardIntRoutine(int CardNumber, InterruptRoutine func);         /* Install User Interrupt Handler Routine */
DLLFUNCW32 int DLLFUNC MCUG3_ResetBoardIntRoutine(int CardNumber);                              /* Deinstall User Interrupt Handler Routine */
DLLFUNCW32 int DLLFUNC MCUG3_SetIntMask(int CardNumber, int i_IRQMaskBits);                     /* Set Interrupt Mask Register */

DLLFUNCW32 void DLLFUNC MsgToScreen (int CardNumber, long Enable);                              /* enable / disable screen messages */
DLLFUNCW32 int  DLLFUNC rdPldVersion(int CardNumber, int *value);                               /* read PLD version number */
DLLFUNCW32 int  DLLFUNC rdOsVersion(int CardNumber, int *value);                                /* read Operation System version number */
DLLFUNCW32 int DLLFUNC wrCmdLine(int CardNumber, char *cmdline);								/* transferring Linux Cmd over rwmos */
DLLFUNCW32 int DLLFUNC wrCmdLineX(int CardNumber, char *cmdline);

DLLFUNCW32 void DLLFUNC wrGCR(int CardNumber, long an, long *value);                            /* write  GCR-Register */
DLLFUNCW32 void DLLFUNC rdGCR(int CardNumber, long an, long *value);                            /* read  GCR-Register */
DLLFUNCW32 void DLLFUNC wrSpecFunc(int CardNumber, long an, long *value);                       /* write  Special-Function-Register */
DLLFUNCW32 void DLLFUNC rdSpecFunc(int CardNumber, long an, long *value);                       /* read  Special-Function-Register */
DLLFUNCW32 void DLLFUNC wrControllerFlags(int CardNumber, long an, long *value);                /* write Controller-Flags */
DLLFUNCW32 void DLLFUNC rdControllerFlags(int CardNumber, long an, long *value);                /* read Controller-Flags */
DLLFUNCW32 int  DLLFUNC CardSelect (int CardNumber);                                            /* Selection with more than one cards in PC */
DLLFUNCW32 void DLLFUNC wrJerkRel (int CardNumber, long an, double *value);                     /* write Jerk-Faktor 0..1 */
DLLFUNCW32 void DLLFUNC rdJerkRel (int CardNumber, long an, double *value);                     /* read  Jerk-Faktor 0..1 */
DLLFUNCW32 void DLLFUNC wrJerk (int CardNumber, long an, double *value);                        /* write Jerk-Value [UU / (TU*TU*TU)] */
DLLFUNCW32 void DLLFUNC rdJerk (int CardNumber, long an, double *value);                        /* read  Jerk-Value [UU / (TU*TU*TU)] */
DLLFUNCW32 void DLLFUNC wrErrorReg(int CardNumber, long *value);                                /* write ErrorRegister */
DLLFUNCW32 void DLLFUNC rdErrorReg(int CardNumber, long *value);                                /* read  ErrorRegister */
DLLFUNCW32 int  DLLFUNC rdSampleTime(int CardNumber, long *SampleTime);                         /* read SampleTime from Card in s */
DLLFUNCW32 void DLLFUNC wrslsp(int CardNumber, long an, double *value);                         /* write Slits-or-Step-Pulses */
DLLFUNCW32 int  DLLFUNC rdslsp(int CardNumber, long an, double *value);                         /* read  Slits-or-Step-Pulses */

DLLFUNCW32 unsigned DLLFUNC rdDriverVersion(int CardNumber);                                    /* rnw Customer Driver Version read */
DLLFUNCW32 unsigned DLLFUNC allocPhysMem (int CardNumber, void **VirtualAdr, unsigned *PhysAdr, unsigned size);
                                                                                                /* allocate physical Memory */
DLLFUNCW32 unsigned DLLFUNC freePhysMem (int CardNumber, void *VirtualAdr);                     /* free physical Memory */
DLLFUNCW32 int DLLFUNC accstop(int CardNumber, struct AS *as);                                  /* acceleration stop in JOG-S-Profiles */
DLLFUNCW32 int DLLFUNC getEnvStr(int CardNumber, char *EnvStr);                                 /* Read Environment String */
DLLFUNCW32 int DLLFUNC gettskinfo(int CardNumber, int TaskNr, int *info);                       /* Read CNC-Task Text Info */
DLLFUNCW32 int DLLFUNC rdNumDigIO(int CardNumber, unsigned int *value);                         /* Dummy Funktion for mcfg */
DLLFUNCW32 int DLLFUNC rddpoffset(int CardNumber, int an, double *value);                       /* read dpoffset */
DLLFUNCW32 int DLLFUNC wrdpoffset(int CardNumber, int an, double *value);                       /* write dpoffset */
DLLFUNCW32 int DLLFUNC rddvoffset(int CardNumber, int an, double *value);                       /* read dvoffset */
DLLFUNCW32 int DLLFUNC wrdvoffset(int CardNumber, int an, double *value);                       /* write dvoffset */
DLLFUNCW32 int DLLFUNC wrtrjerk  (int CardNumber, double *value);                               /* write trajectory jerk */
DLLFUNCW32 int DLLFUNC rdtrjerk  (int CardNumber, double *value);                               /* read trajectory jerk */
DLLFUNCW32 int DLLFUNC wrtrac	 (int CardNumber, double *value);								/* write trajectory acceleration */
DLLFUNCW32 int DLLFUNC rdtrac    (int CardNumber, double *value);								/* read trajectory acceleration */
DLLFUNCW32 int DLLFUNC wrtrvl    (int CardNumber, double *value);								/* write trajectory velocity */
DLLFUNCW32 int DLLFUNC rdtrvl    (int CardNumber, double *value);								/* read trajectory velocity */
DLLFUNCW32 int DLLFUNC wrtrtvl   (int CardNumber, double *value);								/* write trajectory target velocity */
DLLFUNCW32 int DLLFUNC rdtrtvl   (int CardNumber, double *value);								/* read trajectory target velocity */
DLLFUNCW32 int DLLFUNC wrSplineBlockSize(int CardNumber, long *value);                          /* write SplineBlockSize */
DLLFUNCW32 int DLLFUNC rdSplineBlockSize(int CardNumber, long *value);                          /* read  SplineBlockSize */

DLLFUNCW32 int DLLFUNC wrElCamTable64(int CardNumber, char *buffer, int size, int ElCamTableNr);
DLLFUNCW32 int DLLFUNC wrElCamTable32(int CardNumber, char *buffer, int size, int ElCamTableNr);
DLLFUNCW32 int DLLFUNC wrMcug3Handshake(int CardNumber, unsigned *data);

DLLFUNCW32 int DLLFUNC SetOfflineTrace(int CardNumber);                                         /* Activate Offline Trace */
DLLFUNCW32 int DLLFUNC StopTrace(int CardNumber);                                               /* stop Tracing in RWMOS.ELF */

DLLFUNCW32 int DLLFUNC SelectedCard(int CardNumber);                                            /* derzeit selektierte Achskarte abfragen */

DLLFUNCW32 int DLLFUNC rdgfaux(int CardNumber, int an, double *value);                          /* read Gear-Factor Auxiliary */
DLLFUNCW32 int DLLFUNC wrgfaux(int CardNumber, int an, double *value);                          /* write Gear-Factor Auxiliary */

DLLFUNCW32 int DLLFUNC rdUsTimer24b(int CardNumber);                                            /* read MicroSecond Timer 24bit wide */

DLLFUNCW32 void DLLFUNC rdrpsync                (int CardNumber, struct TSRP *tsrp);            /* read real position + sync aux */
DLLFUNCW32 void DLLFUNC rdauxsync               (int CardNumber, struct TSRP *tsrp);            /* read aux syncronized */

DLLFUNCW32 void DLLFUNC ClearCI99               (int CardNumber);                               /* Clear CI99 (for SSF 1025) */
DLLFUNCW32 void DLLFUNC ClearCI               	(int CardNumber, long ndx);                     /* Clear CI?? (for SSF 1026) */
DLLFUNCW32 int  DLLFUNC RW_readmem              (int CardNumber, char *dest, void *source, unsigned size, unsigned offset);
DLLFUNCW32 void DLLFUNC CpuSoftReset			(int CardNumber);

DLLFUNCW32 int DLLFUNC VelToAOut 				(int CardNumber, long an, long TabSize, long NumTab, double *Vmax, void *table);	/* OL Kommando Trajectory Velocity to Analog Out */
DLLFUNCW32 int DLLFUNC SelectTable				(int CardNumber, long an, long TableNdx);		/* Select OL Cmd Table */
DLLFUNCW32 int DLLFUNC rdPcapIndex				(int CardNumber, long an, long *value);			/* Read PcapIndex */

DLLFUNCW32 int DLLFUNC rdSysVarInt				(int CardNumber, long cmd, long *value);		/* Access to Sys- / Ax-Var */
DLLFUNCW32 int DLLFUNC rdSysVarDbl				(int CardNumber, long cmd, double *value);
DLLFUNCW32 int DLLFUNC wrSysVarInt				(int CardNumber, long cmd, long *value);		/* Access to Sys- / Ax-Var */
DLLFUNCW32 int DLLFUNC wrSysVarDbl				(int CardNumber, long cmd, double *value);
DLLFUNCW32 int DLLFUNC rdAxVarInt				(int CardNumber, long cmd, long axndx, long *value);
DLLFUNCW32 int DLLFUNC rdAxVarDbl				(int CardNumber, long cmd, long axndx, double *value);
DLLFUNCW32 int DLLFUNC wrAxVarInt				(int CardNumber, long cmd, long axndx, long *value);
DLLFUNCW32 int DLLFUNC wrAxVarDbl				(int CardNumber, long cmd, long axndx, double *value);
DLLFUNCW32 int DLLFUNC CheckRwmosBusy			(int CardNumber);		/* prfen ob RWMOS gerade in Kommandoausfhrung ist / Hilfsfunktion fr Optimierung des Programmablaufs */
DLLFUNCW32 int DLLFUNC wrFilename				(int CardNumber, char *filename);
DLLFUNCW32 int DLLFUNC ecFileToRead				(int CardNumber, char *filename);
DLLFUNCW32 int DLLFUNC ReqSendFileBlock			(int CardNumber, char *filename);
DLLFUNCW32 int DLLFUNC rdDmesg					(int CardNumber, char *filename);	/* generating DMESG protocol and Transfer to Windows */
DLLFUNCW32 int DLLFUNC wrFile					(int CardNumber, char *filename);
DLLFUNCW32 int DLLFUNC ec_chk_option     		(int CardNumber);                   /* Check for EC Option */

#endif  /* #indef mcug3H */
