/* 	Wrapper Class for McuG3 products
	Rsch & Walter Industrie Elektronik GmbH

    02.10.2017  V2-53-116 mw new function: rdPcapIndex added
    18.03.2016  V2-53-111 mw new functions: wrShortProfile and rdShortProfile
    18.02.2016  V2-53-110 mw new functions: rddigof, rdpeaux, rdVBErr, InitMcuSystem4
                                            wrtrac, wrtrvl, wrtrtvl, rdtrac, rdtrvl, rdtrtvl
 	15.03.2011	V2-53-79 mw	new function RPtoDP
    20.05.2009  V2-53-71 mw new function ClearCI99
	04/01/2008	V2-53-57 mw	new functions rdgfaux and wrgfaux
	12.12.2007	V2-53.56 mw	rdifs now returns zero, if function sucessfully has been executet
							up to now the function hasn't any return value
    10.12.2007 V2-53.56 mw  New Functions wrtrovrst, rdrtrovrst added
							for handling of trovr settling time
    2007-06-20 mw	created
*/

using System;
using System.Runtime.InteropServices;
using System.Collections.Generic;
using System.Text;

namespace MCUG3
{
	public  class MCUG3Card   /* Klassenkonstruktor */
    {
        #region Variables
        public TSRP[]       tsrp;
        public LMP          lmp;
        public TRU          tru;
        public ROSI         rosi;
        public BOARDTYPE    BoardType;
        public CIBUF        ci;
        public CDBUF        cd;
        public AS           as_;
        public CMP          cmp;
        public HMP          hmp;
        public HMP3D        hmp3d;
        public CNCTS        cncts;
        public CBCNCT[]     cbcnct;
        public CTRLRPARAMS  ctrlrparams;
        public LATCH_INFOS  latch_infos;

		public enum BOARDTYPE : int
		{
			/// <value></value>
			MCU3000 = 4,
			/// <value></value>
			MCU6000 = 8
        }
        #endregion

        #region Constructor, Destructor

        public MCUG3Card()
		{
			tsrp            = new TSRP[MAXAXIS];
            ci.civals       = new int[SIZE_CIBUF];
            cd.cdvals       = new double[SIZE_CDBUF];
            as_.san         = new int[MAXAXIS];
            lmp.dtm         = new double[MAXAXIS];
            hmp.dtm         = new double[MAXAXIS];
            hmp3d.dtma      = new double[MAXAXIS];
            ctrlrparams.val = new double[SCPSZ_R * SCPSZ_C];
        }


		#endregion

        #region DLL Structure-Defines

        internal const int MAXAXIS     = 18; 			/* currently maximum supplied number of axis, do not change !!! */
        private  const int SIZE_STRREV = 40;			/* String-Size definition */
        private  const int SIZE_CIBUF  = 1000;			/* Number of elements in common integer buffer */
        private  const int SIZE_CDBUF  = 1000;			/* Number of elements in common double buffer */
        internal const int SCPSZ_R     = 15;	  	    /* row size for CTRLRPARAMS */
        internal const int SCPSZ_C     = 15;			/* colomn size for CTRLRPARAMS*/

        [StructLayout(LayoutKind.Explicit)]
        public struct AXST
        {
            [FieldOffset(0)]
            public int status_word;
            /// <summary>
            ///	axst: Error- + Status-Bits
            ///	unsigned toasm  : 1;			/* Bit 0:  MCU-6000 only, MCU-3000 / APCI-8001 = 0 */
            ///	unsigned eo     : 1;			/* Bit 1:  Emergency out */
            ///	unsigned dnr    : 1;			/* Bit 2:  Drive not ready */
            ///	unsigned lslh	: 1;			/* Bit 3:  Limit switch left hardware */
            ///	unsigned lsrh	: 1;			/* Bit 4:  Limit switch right hardware */
            ///	unsigned lsls   : 1;			/* Bit 5:  Limit switch left software */
            ///	unsigned lsrs   : 1;			/* Bit 6:  Limit switch right software */
            ///	unsigned mpe	: 1; 			/* Bit 7:  Maximum Position Error */
            ///	unsigned dhef	: 1;			/* Bit 8:  data handling error flag */
            ///	unsigned cef	: 1;			/* Bit 9:  configuration error flag */
            ///	unsigned nc	    : 2;			/* Bit 10..11: currently not used */
            ///
            ///	/* Status-Bits */
            ///	unsigned pe	    : 1;			/* Bit 12: Profile end */
            ///	unsigned cl	    : 1;			/* Bit 13: Closed loop */
            ///	unsigned ip	    : 1;			/* Bit 14: In Position */
            ///	unsigned ui	    : 1;			/* Bit 15: user input */
            ///	unsigned lpsf	: 1;			/* Bit 16: latched position synchronous flag */
            /// unsigned ar		: 1;            /* Bit 17: axis referenced flag */
            ///	unsigned temp2	: 13; 			/* Bit 18..29: currently not used */
            ///	  
            ///	unsigned 		: 2;			/* Bit 30..31: for internal use only */
            /// </summary>
            [FieldOffset(0)]
            public int bit;
        };

        [StructLayout(LayoutKind.Sequential)]
        public struct ROSI
        {

            [MarshalAs(UnmanagedType.ByValArray, SizeConst = SIZE_STRREV)]
            public char[] revision;

            public int Number_Axis;
            public int SysFile_Loaded;
        };

        [StructLayout(LayoutKind.Sequential)]
        public struct CIBUF
        {
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = SIZE_CIBUF)]
            public int[] civals;
        };

        [StructLayout(LayoutKind.Sequential)]
        public struct CDBUF
        {
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = SIZE_CDBUF)]
            public double[] cdvals;
        };

        [StructLayout(LayoutKind.Explicit)]
        public struct TSRP
        {
            [FieldOffset(0)]   public int an;
            [FieldOffset(4)]   public double kp;
            [FieldOffset(12)]  public double ki;
            [FieldOffset(20)]  public double kd;
            [FieldOffset(28)]  public double kpl;
            [FieldOffset(36)]  public double kfca;
            [FieldOffset(44)]  public double kfcv;
            [FieldOffset(52)]  public double jac;
            [FieldOffset(60)]  public double jvl;
            [FieldOffset(68)]  public double jtvl;
            [FieldOffset(76)]  public double jovr;
            [FieldOffset(84)]  public double hac;
            [FieldOffset(92)]  public double hvl;
            [FieldOffset(100)] public double rp;
            [FieldOffset(108)] public double dp;
            [FieldOffset(116)] public double tp;
            [FieldOffset(124)] public double dv;
            [FieldOffset(132)] public double lp;
            [FieldOffset(140)] public double aux;
            [FieldOffset(148)] public double sll;
            [FieldOffset(156)] public double slr;
            [FieldOffset(164)] public double ipw;
            [FieldOffset(172)] public double mpe;
            [FieldOffset(180)] public double sdec;
            [FieldOffset(188)] public double gf;
            [FieldOffset(196)] public int mcp;
            [FieldOffset(200)] public AXST axst;
            [FieldOffset(204)] public int lsm;
            [FieldOffset(208)] public int epc;
            [FieldOffset(212)] public int digi;
            [FieldOffset(216)] public int digo;
            [FieldOffset(220)] public int ifs;
            /// <summary>
            /// dummy...: reserved space don't delete!!
            /// </summary>
            [FieldOffset(224)] public int dummy1;
            [FieldOffset(228)] public int dummy2;
            [FieldOffset(232)] public int dummy3;
            [FieldOffset(236)] public int dummy4;
        };

        [StructLayout(LayoutKind.Explicit)]
        public struct LMP
        {
            public LMP(double ac, double vl, double tvl)
            {
                this.ac = ac;
                this.vl = vl;
                this.tvl = tvl;
                this.dtm = new double[MAXAXIS];
            }
            [FieldOffset(0)]  public double ac;
            [FieldOffset(8)]  public double vl;
            [FieldOffset(16)] public double tvl;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = MAXAXIS)]
            [FieldOffset(24)] public double[] dtm;
        };

        [StructLayout(LayoutKind.Explicit)]
        public struct CMP
        {
            public CMP(double ac, double vl, double tvl, double phi, double dtca1, double dtca2)
            {
                this.ac = ac;
                this.vl = vl;
                this.tvl = tvl;
                this.phi = 0.0;
                this.dtca1 = 0.0;
                this.dtca2 = 0.0;
            }
            [FieldOffset(0)]
            public double ac;
            [FieldOffset(8)]
            public double vl;
            [FieldOffset(16)]
            public double tvl;
            [FieldOffset(24)]
            public double phi;
            [FieldOffset(32)]
            public double dtca1;
            [FieldOffset(40)]
            public double dtca2;
        };

        [StructLayout(LayoutKind.Explicit)]
        public struct HMP
        {
            public HMP(double ac, double vl, double tvl, double phi, double dtca1, double dtca2)
            {
                this.ac     = ac;
                this.vl     = vl;
                this.tvl    = tvl;
                this.phi    = phi;
                this.dtca1  = dtca1;
                this.dtca2  = dtca2;
                this.dtm    = new double[MAXAXIS];
            }
            [FieldOffset(0)]
            public double ac;
            [FieldOffset(8)]
            public double vl;
            [FieldOffset(16)]
            public double tvl;
            [FieldOffset(24)]
            public double phi;
            [FieldOffset(32)]
            public double dtca1;
            [FieldOffset(40)]
            public double dtca2;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = MAXAXIS)]
            [FieldOffset(48)]
            public double[] dtm;
        };

        [StructLayout(LayoutKind.Explicit)]
        public struct HMP3D
        {
            public HMP3D(double ac, double vl, double tvl)
            {
                this.ac = ac;
                this.vl = vl;
                this.tvl = tvl;
                this.phi = 0.0;
                this.dtca1 = 0.0;
                this.dtca2 = 0.0;
                this.dtca3 = 0.0;
                this.pn1 = 0.0;
                this.pn2 = 0.0;
                this.pn3 = 0.0;
                this.dtma = new double[MAXAXIS];
            }
            [FieldOffset(0)]
            public double ac;
            [FieldOffset(8)]
            public double vl;
            [FieldOffset(16)]
            public double tvl;
            [FieldOffset(24)]
            public double phi;
            [FieldOffset(32)]
            public double dtca1;
            [FieldOffset(40)]
            public double dtca2;
            [FieldOffset(48)]
            public double dtca3;
            [FieldOffset(56)]
            public double pn1;
            [FieldOffset(64)]
            public double pn2;
            [FieldOffset(72)]
            public double pn3;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = MAXAXIS)]
            [FieldOffset(80)]
            public double[] dtma;
        };

        [StructLayout(LayoutKind.Sequential)]
		public struct AS 
		{
			public AS(int unoa,int[] san)
			{
				this.unoa=unoa;
				this.san=new int[MAXAXIS];
				for (int i=0; i<unoa; i++)
				{
					this.san[i]=san[i];
				}
			}
			public AS(int san)
			{
				this.unoa=1;
				this.san=new int[MAXAXIS];
				this.san[0]=san;
			}

			public int					unoa;
			[MarshalAs(UnmanagedType.ByValArray, SizeConst=MAXAXIS)]    
			public int[]				san;										
		};

        [StructLayout(LayoutKind.Sequential)]
        public struct TRU
        {
            public int pu;      // Position Unit
            public int tu;      // Time Unit
        };

        [StructLayout(LayoutKind.Sequential)]
        public struct CNCTS
        {
            public int ErrNum;
            public int ErrLine;
            public int StackFree;
            public int Running;
            public int csrcline;
        };

        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
        public struct CBCNCT
        {
            public int taskNr;
            public int size;
            public IntPtr buffer;
        };

        [StructLayout(LayoutKind.Sequential)]
        public struct CTRLRPARAMS
        {
            public CTRLRPARAMS(int defaultValue)
            {
                this.val = new double[SCPSZ_R * SCPSZ_C];
                for (int i = 0; i < SCPSZ_R * SCPSZ_C; i++)
                    this.val[i] = defaultValue;
            }
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = SCPSZ_R * SCPSZ_C)]
            public double[] val;
        };

        [StructLayout(LayoutKind.Explicit)]
        public struct OptionDescriptorObject
        {
            public OptionDescriptorObject(UInt32 AccessType, UInt32 DataType, UInt32 BusNumber, UInt32 DeviceNumber, UInt32 Index, UInt32 SubIndex)
            {
                this.Handle         = 0;
                this.AccessType     = AccessType;
                this.DataType       = DataType;
                this.BusNumber      = BusNumber;
                this.DeviceNumber   = DeviceNumber;
                this.Index          = Index;
                this.SubIndex       = SubIndex;
            }
            [FieldOffset(0)]  public UInt32 Handle;
            [FieldOffset(4)]  public UInt32 AccessType;
            [FieldOffset(8)]  public UInt32 DataType;
            [FieldOffset(12)] public UInt32 BusNumber;
            [FieldOffset(16)] public UInt32 DeviceNumber;
            [FieldOffset(20)] public UInt32 Index;
            [FieldOffset(24)] public UInt32 SubIndex;
        };

        [StructLayout(LayoutKind.Explicit)]
        public struct LATCH_INFOS
        {
            public LATCH_INFOS(int san, double lst, double let, int pu, int tu)
            {
                this.san = san;
                this.lst = lst;
                this.let = let;
                this.pu = pu;
                this.tu = tu;
            }
            [FieldOffset(0)]
            public int san;
            [FieldOffset(4)]
            public double lst;
            [FieldOffset(12)]
            public double let;
            [FieldOffset(20)]
            public int pu;
            [FieldOffset(24)]
            public int tu;
        };

#endregion

        #region Private Functions
        /// <summary>
		/// intial the MCU system
		/// Diese Funktion fhrt die komplette Software-Initialisierung des Antriebssystems durch. 
		/// Der Funktionsaufruf mu zu Beginn (in jedem Fall vor anderen PCAPAufrufen)
		/// in jedem PCAP-Anwenderprogramms ausgefhrt werden. 
		/// Innerhalb dieser Funktion werden verschiedene PCAP-Basis-Funktionen aufgerufen. 
		/// Unter anderem werden die Achsennummern {an} in der Struktur tsrp initialisiert. 
		/// Sofern die Systemdatei system.dat noch nicht auf die MCU-G3 bertragen wurde, 
		/// wird dies hier getan. 
		/// Am Ende der Funktion werden die Achsparameter von allen	Achsen in die Struktur tsrp eingelesen.
		/// </summary>
		/// <param name="tpuBaseAddress"></param>
		/// <param name="bootFileName"></param>
		/// <param name="systemFileName">enthlt den Dateinamen der Systemdatei (normalerweise system.dat) inkl. Pfad- und Laufwerksangabe.</param>
		/// <param name="boardType">Der Wert von BoardType gibt Aufschluss ber den Steuerungstyp.
		/// BoardType 4 es handelt es sich um eine MCU-3000 oder APCI-8001
		/// BoardType 8 es handelt es sich um eine MCU-6000 oder APCI-8401.
		/// </param>
		/// <returns>
		///	    0 Kein Fehler.
		///    31 Es wurde kein MCU-G3-Controller gefunden.
		///    32 Die rw_MOS-Betriebssystemsoftware ist nicht geladen oder wurde gestoppt. Siehe hierzu PCAP-Kommando BootFile() oder Dienstprogramm mcbt.exe
		///    33 Falsche Betriebssystemsoftware. Die Dateiversionen der	 mcug3.dll und rwmos.elf-Files haben inkompatible Revisionsstnde und knnen nicht aufeinander abgestimmt	werden.
		///    34 Der Gertetreiber rnwmc.sys (Windows NT 4.0, 2000) oder rnwmc.vxd (Windows 95/98/Me) konnte nicht geffnet werden.
		///    35 Fehler beim Einmappen des physischen MCU-G3-Speichers.
		///    36 Fehler beim Einmappen des physischen MCU-G3-Speichers.
		///    37 Fehler beim Ausmappen des physischen MCU-G3-Speichers.
		///    38 Zugriff auf die MCU-G3 nicht mglich.
		///    39 Kein Zugriff auf MCU-G3 Mail-Box-Interface mglich.
		/// lderr Fehler-Rckgabewert von PCAP-Befehl txbf()Die Funktion hat die gleichen Rckgabewerte wie die Funktion InitMcuSystem(). 
		/// 
		/// Zustzlich wird die Struktur ROSI anhand der von der Steuerung zurckgelieferten Systeminformationen aktualisiert. 
		///</returns>
		/// <remarks>
		/// Es wird die Systemdatei system.dat, welche hauptschlich mit Hilfe des TOOLSET-Programms mcfg.exe verndert wird,
		/// auf die MCU-G3 bertragen und bewirkt dort die Initialisierung systeminterner Parameter wie z.B.
		/// Beschleunigungen, Geschwindigkeiten, Filterkoeffizienten, Grenzwerte usw. Dieser Ladevorgang muss
		///  einmalig pro Systemstart erfolgen.</remarks>
		internal int InitSystem(int TpuBaseAddress, 
			string bootFileName, 
			string systemFileName, 
			ref BOARDTYPE BoardType)
		{
			int mcuBoardType=0;
			//intial the MCU system
			int error= InitMcuSystem3(tsrp, ref rosi, TpuBaseAddress, systemFileName, ref mcuBoardType);
			if (error != 0)
			{
				//boot CNC system
				error=BootFile(bootFileName, TpuBaseAddress);
				error= InitMcuSystem3(tsrp, ref rosi, TpuBaseAddress, systemFileName, ref mcuBoardType);
			}

            BoardType = (BOARDTYPE)mcuBoardType;
	
			return error;
		}
		#endregion

        #region C-DLL function calls

        [DllImport("mcug3.dll")]
        public static extern void uf(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void ctru(
            ref TRU tru);

        [DllImport("mcug3.dll")]
        public static extern void wrjac(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrjvl(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrjtvl(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrhac(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrhvl(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrrp(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrdp(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrlp(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wraux(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void shp(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrdigo(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrdigob(
            int an, int bitnr, int value);

        [DllImport("mcug3.dll")]
        public static extern void wrasmo(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrasmob(
            int an, int bitnr, int value);

        [DllImport("mcug3.dll")]
        public static extern void rdrp(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rddp(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdtp(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rddv(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdlp(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdaux(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rddigi(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern int rddigib(
            int an, int bitnr);

        [DllImport("mcug3.dll")]
        public static extern int rdifs(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern int rdifsb(
            int an, int bitnr);

        [DllImport("mcug3.dll")]
        public static extern void rddigo(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rddigof(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern int rddigob(
            int an, int bitnr);

        [DllImport("mcug3.dll")]
        public static extern void rdasmi(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern int rdasmib(
            int an, int bitnr);

        [DllImport("mcug3.dll")]
        public static extern void rdasms(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern int rdasmsb(
            int an, int bitnr);

        [DllImport("mcug3.dll")]
        public static extern void rdepc(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdasmepc(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdasmo(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern int rdasmob(
            int an, int bitnr);

        [DllImport("mcug3.dll")]
        public static extern void rdlsm(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdaxst(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern int rdaxstb(
            int an, int bitnr);

        [DllImport("mcug3.dll")]
        public static extern void rdjac(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdjvl(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdjtvl(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdhac(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdhvl(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void cl(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern void clv(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern void ol(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern void ra(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern void rifs(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rasms(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rs();

        [DllImport("mcug3.dll")]
        public static extern void azo(
            int set);

        [DllImport("mcug3.dll")]
        public static extern void jr(
            ref AS as_,
            [In, Out] TSRP[] tsrp);

        [DllImport("mcug3.dll")]
        public static extern void ja(
            ref AS as_,
            [In, Out] TSRP[] tsrp);

        [DllImport("mcug3.dll")]
        public static extern void jhl(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern void jhr(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern void jhi(
            ref AS as_,
            [In, Out] TSRP[] tsrp);

        [DllImport("mcug3.dll")]
        public static extern void mlr(
            ref AS as_,
            ref LMP lmp);

        [DllImport("mcug3.dll")]
        public static extern void mla(
            ref AS as_,
            ref LMP lmp);

        [DllImport("mcug3.dll")]
        public static extern void mcr(
            ref AS as_,
            ref CMP cmp);

        [DllImport("mcug3.dll")]
        public static extern void mca(
            ref AS as_,
            ref CMP cmp);

        [DllImport("mcug3.dll")]
        public static extern void mhr(
            ref AS as_,
            ref CMP hmp);

        [DllImport("mcug3.dll")]
        public static extern void mha(
            ref AS as_,
            ref CMP hmp);

        [DllImport("mcug3.dll")]
        public static extern void mca3d(
            ref AS as_,
            ref CMP hmp3d);

        [DllImport("mcug3.dll")]
        public static extern void mcr3d(
            ref AS as_,
            ref CMP hmp3d);

        [DllImport("mcug3.dll")]
        public static extern void smca3d(
            ref AS as_,
            ref CMP hmp3d);

        [DllImport("mcug3.dll")]
        public static extern void smcr3d(
            ref AS as_,
            ref CMP hmp3d);

        [DllImport("mcug3.dll")]
        public static extern void smlr(
            ref AS as_,
            ref LMP lmp);

        [DllImport("mcug3.dll")]
        public static extern void smla(
            ref AS as_,
            ref LMP lmp);

        [DllImport("mcug3.dll")]
        public static extern void smcr(
            ref AS as_,
            ref CMP cmp);

        [DllImport("mcug3.dll")]
        public static extern void smca(
            ref AS as_,
            ref CMP cmp);

        [DllImport("mcug3.dll")]
        public static extern void smhr(
            ref AS as_,
            ref CMP hmp);

        [DllImport("mcug3.dll")]
        public static extern void smha(
            ref AS as_,
            ref CMP hmp);

        [DllImport("mcug3.dll")]
        public static extern void ssms(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern void sstps(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern void sdels(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern int txbf2(
            [In, MarshalAs(UnmanagedType.LPStr)] string filename);

        [DllImport("mcug3.dll")]
        public static extern void startcnct(
            [In] int TaskNr);

        [DllImport("mcug3.dll")]
        public static extern void stopcnct(
            [In] int TaskNr);

        [DllImport("mcug3.dll")]
        public static extern void contcnct(
            [In] int TaskNr);

        [DllImport("mcug3.dll")]
        public static extern void stepcnct(
            [In] int TaskNr);

        [DllImport("mcug3.dll")]
        public static extern void rdcncts(
            [In] int TaskNr,
            ref CNCTS cncts);

        [DllImport("mcug3.dll")]
        public static extern int rdcbcnct(
            ref CBCNCT cbcnct);

        [DllImport("mcug3.dll")]
        public static extern int wrcbcnct(
            ref CBCNCT cbcnct);

        [DllImport("mcug3.dll")]
        public static extern void rdci(
            [In] int ndx,
            ref CIBUF ci);

        [DllImport("mcug3.dll")]
        public static extern void rdcd(
            [In] int ndx,
            ref CDBUF cd);

        [DllImport("mcug3.dll")]
        public static extern void wrci(
            [In] int ndx,
            ref CIBUF ci);

       [DllImport("mcug3.dll")] 
       public static extern void wrcd(
            [In] int ndx,
            ref CDBUF cd);

        [DllImport("mcug3.dll")]
        public static extern void rdmcp(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrmcp(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrledrd(
             [In] int value);

        [DllImport("mcug3.dll")]
        public static extern void wrledyl(
             [In] int value);

        [DllImport("mcug3.dll")]
        public static extern void wrledgn(
             [In] int value);

        [DllImport("mcug3.dll")]
        public static extern int rdledrd();

        [DllImport("mcug3.dll")]
        public static extern int rdledyl();

        [DllImport("mcug3.dll")]
        public static extern int rdledgn();

        [DllImport("mcug3.dll")]
        public static extern int rdirqpc();

        [DllImport("mcug3.dll")]
        public static extern int mcuinit(
            ref ROSI rosi);

        [DllImport("mcug3.dll")]
        public static extern int mcuinit2(
            ref ROSI rosi,
            [In] int TpuBaseAddress);

        [DllImport("mcug3.dll")]
        public static extern void wrjovr(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdjovr(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrtrovr(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void rdtrovr(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int wrtrovrst(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdtrovrst(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int wrShortProfile(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdShortProfile(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void utrovr(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern void rdf(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdap(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdsll(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrsll(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdslr(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrslr(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdipw(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wripw(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdmpe(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrmpe(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdsdec(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrsdec(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrgf(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdgf(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void ms(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern void js(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern void lps(
            int an,
            int mst);

        [DllImport("mcug3.dll")]
        public static extern void rdigi(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdlpndx(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void wrlpndx(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void ssf(
            int an,
            int command,
            double value);

        [DllImport("mcug3.dll")]
        public static extern void ssfn(
            ref AS as_,
            int command,
            double value);

        [DllImport("mcug3.dll")]
        public static extern void InitMcuSystem(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern int InitMcuSystem2(
            [In, Out] TSRP[] tsrp,
            [In] int TpuBaseAddress,
            [In, MarshalAs(UnmanagedType.LPStr)] string SystemFileName);

        [DllImport("mcug3.dll")]
        public static extern int InitMcuSystem3(
            [In, Out] TSRP[] tsrp,
            ref ROSI rosi,
            [In] int TpuBaseAddress,
            [In, MarshalAs(UnmanagedType.LPStr)] string SystemFileName,
            [In, Out, MarshalAs(UnmanagedType.I4)] ref int BoardType);

        [DllImport("mcug3.dll")]
        public static extern int InitMcuSystem4(
            [In, Out] TSRP[] tsrp,
            ref ROSI rosi,
            [In] int TpuBaseAddress,
            [In, MarshalAs(UnmanagedType.LPStr)] string SystemFileName,
            [In, Out, MarshalAs(UnmanagedType.I4)] ref int BoardType,
            [In, Out, MarshalAs(UnmanagedType.I4)] ref int NrCards);

        [DllImport("mcug3.dll")]
        public static extern void InitMcuErrorReport(
            int error);

        [DllImport("mcug3.dll")]
        public static extern void txbfErrorReport(
            [In, MarshalAs(UnmanagedType.LPStr)] string filename,
            int error);

        [DllImport("mcug3.dll")]
        public static extern void BootErrorReport(
            [In, MarshalAs(UnmanagedType.LPStr)] string filename,
            int error);

        [DllImport("mcug3.dll")]
        public static extern void McuClose();

        [DllImport("mcug3.dll")]
        public static extern int BootFile(
            [In, MarshalAs(UnmanagedType.LPStr)] string BootFileName,
            [In] int TpuBaseAddress);

        [DllImport("mcug3.dll")]
        public static extern void LdSSplinea(
            ref AS as_,
            ref LMP lmp);

        [DllImport("mcug3.dll")]
        public static extern void LdSSplineR(
            ref AS as_,
            ref LMP lmp);

        [DllImport("mcug3.dll")]
        public static extern void ExeSSpline(
            ref AS as_,
            ref LMP lmp);

        [DllImport("mcug3.dll")]
        public static extern int wrOptionInt(
            ref OptionDescriptorObject odesc,
            ref int val);

        [DllImport("mcug3.dll")]
        public static extern int rdOptionInt(
            ref OptionDescriptorObject odesc,
            ref int val);

        [DllImport("mcug3.dll")]
        public static extern int wrOptionDbl(
            ref OptionDescriptorObject odesc,
            ref double val);

        [DllImport("mcug3.dll")]
        public static extern int rdOptionDbl(
            ref OptionDescriptorObject odesc,
            ref double val);

        [DllImport("mcug3.dll")]
        public static extern int freeOptionHandle(
            ref OptionDescriptorObject odesc);

        [DllImport("mcug3.dll")]
        public static extern void wrModeReg(
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void rdModeReg(
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void wrnfrax(
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void rdnfrax(
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void wrMDVel(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void rdMDVel(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void wrMaxVel(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void rdMaxVel(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void wrMaxAcc(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void rdMaxAcc(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void wrEffRadius(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void rdEffRadius(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void rdPosErr(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void rdMCiS(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdZeroOffset(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdptp(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdpir(
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern int rddtv(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdScannerBufferSize();

        [DllImport("mcug3.dll")]
        public static extern int rdScannerStatus();

        [DllImport("mcug3.dll")]
        public static extern int rdScannerLsm();

        [DllImport("mcug3.dll")]
        public static extern void wrtp(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdrpd(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rddpd(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void rdtpd(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void scp(
            int an,
            ref CTRLRPARAMS ctrlrparams);

        [DllImport("mcug3.dll")]
        public static extern void rdrv(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void cm(
            int an,
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void getcnctsrclinenr(
            int TaskNr,
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void burnaplpar(
            int an);

        [DllImport("mcug3.dll")]
        public static extern void gettimers(
            ref TSRP tsrp,
            int an);

        [DllImport("mcug3.dll")]
        public static extern void lpr(
            ref LATCH_INFOS latch_infos);

        [DllImport("mcug3.dll")]
        public static extern void lprs(
            ref AS as_,
            ref LATCH_INFOS latch_infos);

        [DllImport("mcug3.dll")]
        public static extern void rdasmecs(
            ref TSRP tsrp);

        [DllImport("mcug3.dll")]
        public static extern void ReadLinkStatus(
            ref int fae,
            ref int ba);

        [DllImport("mcug3.dll")]
        public static extern int rdAin(
            int an,
            int channel);

        [DllImport("mcug3.dll")]
        public static extern void szpa(
            int an,
            double Position);

        [DllImport("mcug3.dll")]
        public static extern void szpr(
            int an,
            double Position);

        [DllImport("mcug3.dll")]
        public static extern int MCUG3_ResetBoardIntRoutine();

        [DllImport("mcug3.dll")]
        public static extern int MCUG3_SetIntMask(
            int i_IRQMaskBits);

        [DllImport("mcug3.dll")]
        public static extern void MsgToScreen(
            int value);

        [DllImport("mcug3.dll")]
        public static extern int rdPldVersion(
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern int rdOsVersion(
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void wrGCR(
            int an,
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void rdGCR(
            int an,
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void wrSpecFunc(
            int an,
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void rdSpecFunc(
            int an,
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void wrControllerFlags(
            int an,
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void rdControllerFlags(
            int an,
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern int CardSelect(
            int CardNumber);

        [DllImport("mcug3.dll")]
        public static extern void wrJerkRel(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void rdJerkRel(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void wrJerk(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void rdJerk(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void wrErrorReg(
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern void rdErrorReg(
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern int rdSampleTime(
            ref int SampleTime);

        [DllImport("mcug3.dll")]
        public static extern void wrslsp(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdslsp(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int gettskinfo(
            int TaskNr,
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern int rddpoffset(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int wrdpoffset(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int wrtrjerk(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdtrjerk(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern UInt32 rdDriverVersion();

        [DllImport("mcug3.dll")]
        public static extern int accstop(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern int rdNumDigIO(
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern int wrSplineBlockSize(
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern int rdSplineBlockSize(
            ref int value);

        [DllImport("mcug3.dll")]
        public static extern int rdgfaux(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int wrgfaux(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void ClearCI99(
            );

        [DllImport("mcug3.dll")]
        public static extern void RPtoDP(
            ref AS as_);

        [DllImport("mcug3.dll")]
        public static extern int wrtrac(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdtrac(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int wrtrvl(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdtrvl(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int wrtrtvl(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdtrtvl(
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdpeaux(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int rdVBError(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern int wrVBError(
            int an,
            ref double value);

        [DllImport("mcug3.dll")]
        public static extern void rdPcapIndex(
            int an,
            ref int value);



#endregion

        internal int InitMcuSystem3()
        {
            throw new Exception("The method or operation is not implemented.");
        }
    }
}


